/*
* Zero-Clause BSD
*
* Permission to use, copy, modify, and/or distribute this software for any purpose with or without fee is hereby granted.
*
* THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED
* WARRANTIES OF MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY SPECIAL, DIRECT, INDIRECT, OR
* CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
* NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
*/

#if ((defined _WINDOWS) || (defined WIN32) || (defined WIN64))
    #include <Windows.h>    // Standard windows includes
#endif // ((defined _WINDOWS) || (defined WIN32) || (defined WIN64))

#include "ThirdPartyComponent5V.h"  // Me self...
#include <stdio.h>                  // fopen, fclose, fwrite
#include <new>                      // new
#include <assert.h>                 // assert()
#include <shlobj.h>                 // SHGetSpecialFolderPath
#include <time.h>                   // localtime_s, time_t
#include "RFC1144.h"                // CRFC1144

// ------------------------------------------------------------------------------------------------------
#if !((defined _WINDOWS) || (defined WIN32) || (defined WIN64))
    // Define POSIX versions
    #define wcsnlen_s           wcsnlen
    #define wcsncpy_s           wcsncpy
    #define strncpy_s           strncpy
    #define swprintf_s          swprintf
#endif // !((defined _WINDOWS) || (defined WIN32) || (defined WIN64))

// Macros and function for secure add text to a buffer
#ifndef min
    #define __Sec_Min( a, b )                                       (((a) < (b)) ? (a) : (b))
#else  // min
    #define __Sec_Min                                               min
#endif // min
#ifndef max
    #define __Sec_Max( a, b )                                       (((a) > (b)) ? (a) : (b))
#else  // min
    #define __Sec_Max                                               max
#endif // min

// Clean string
#define __Sec_Empty( sDestString )                                  (sDestString)[0] = 0

// Size of the string
#define __Sec_sizeof( sDestString )                                 ((size_t)sizeof(sDestString)/(size_t)sizeof((sDestString)[0]))
#define __Sec_sizeofN( sDestString, nMaxLength )                    __Sec_Min(__Sec_sizeof(sDestString), nMaxLength)

// Force to terminate string
#define __Sec_CloseN( sDestString, nMaxLength )                     sDestString[__Sec_Max(__Sec_sizeofN(sDestString,nMaxLength),1)-1] = 0;

// Give pack string length
#define __Sec_Strnlen( sDestString, nMaxLength )                    wcsnlen_s( sDestString, __Sec_sizeofN(sDestString, nMaxLength) )

// Give back remaining string length
#define __Sec_Remaining_strlen( sDestString, nMaxLength )           (__Sec_sizeofN(sDestString, nMaxLength) - __Sec_Strnlen(sDestString, nMaxLength))

// Add new formarted string
#define __Sec_Add_sprintf( sDestString, nMaxLength, sFormat, ... )  if ( (1 < __Sec_Remaining_strlen(sDestString, nMaxLength)) )\
                                                                    {   (void)swprintf_s( &sDestString[__Sec_Strnlen(sDestString, nMaxLength)], __Sec_Remaining_strlen(sDestString, nMaxLength), sFormat, __VA_ARGS__ );    }\
                                                                    __Sec_CloseN( sDestString, nMaxLength )

// Add static string with defined length
#define __Sec_strncpy( sDestString, sSrcString, nMaxLength )        (void)wcsncpy_s( sDestString, sSrcString , __Sec_sizeofN(sDestString, nMaxLength) );\
                                                                    __Sec_CloseN( sDestString, nMaxLength )
#define __Sec_strncpy2( sDestString, sSrcString, nMaxLength )       (void)wcsncpy_s( sDestString, nMaxLength, sSrcString, _TRUNCATE );\
                                                                    (sDestString)[__Sec_Max(1,nMaxLength)-1] = 0

// Add static string (until maximum allowed length)
#define __Sec_strcpy( sDestString, sSrcString )                     __Sec_strncpy( sDestString, sSrcString, __Sec_sizeof(sDestString) )

// Add static ASC string with defined length
#define __Sec_strncpy_asc( sDestString, sSrcString, nMaxLength )    strncpy_s( sDestString, sSrcString , __Sec_sizeofN(sDestString, nMaxLength) );\
                                                                    __Sec_CloseN( sDestString, nMaxLength )

// Add static ASC string (until maximum allowed length)
#define __Sec_strcpy_asc( sDestString, sSrcString )                 __Sec_strncpy_asc( sDestString, sSrcString, __Sec_sizeof(sDestString) )

// Macros and function for secure add text to a buffer
#define __Sec_strrchr( sSourceA, nChar )                            ((intptr_t)((wcsrchr( sSourceA, nChar ))-(const wchar_t*)sSourceA))
// ------------------------------------------------------------------------------------------------------


// ------------------------------------------------------------------------------------------------------
typedef enum EnumResultFilterSettings : unsigned long long
{ // Values for "StructThirdPartyComponent5V_Result::ullFilterMask"
    Filter_ViewAll                                  = 0,
    // ------------------------------------------------------------------------------------------------------------------------------------------------
    TetraTmo_Supress_Slot_1                         = 0x0000000000000001,   // Supress timeslot
    TetraTmo_Supress_Slot_2                         = 0x0000000000000002,   // Supress timeslot
    TetraTmo_Supress_Slot_3                         = 0x0000000000000004,   // Supress timeslot
    TetraTmo_Supress_Slot_4                         = 0x0000000000000008,   // Supress timeslot

    TetraTmo_Supress_DL_NullPdu                     = 0x0000000000000010,   // Suppress NULL-PDU as end of data in PDU
    TetraTmo_Supress_DL_EqualBroadCast              = 0x0000000000000020,   // SYNC, SYSINFO(-Q/DA), ACCESS-DEFINE
    TetraTmo_Supress_DL_EqualDnwrk                  = 0x0000000000000040,   // D-NWRK(-DA), D-NWRK Extension
    TetraTmo_Supress_DL_NotDecoded                  = 0x0000000000000080,   // Supress physical layer information
    TetraTmo_Supress_DL_USignal                     = 0x0000000000000100,   // Supress MAC-U-SIGNAL on FACCH
    TetraTmo_Supress_DL_ErrorInPdu                  = 0x0000000000001000,   // Supress PDUs if an error occured

    TetraTmo_Supress_UL_NullPdu                     = 0x0000000000000200,   // Suppress NULL-PDU as end of data in PDU (not stand alone NULL-PDU!)
    TetraTmo_Supress_UL_NotDecodedBelowThreshhold   = 0x0000000000000400,   // Supress physical layer information, if they are below RSSI treshhold
    TetraTmo_Supress_UL_USignal                     = 0x0000000000000800,   // Supress MAC-U-SIGNAL on FACCH
    TetraTmo_Supress_UL_ErrorInPdu                  = 0x0000000000002000,   // Supress PDUs if an error occured

    TetraTmo_Supress_UserDefined_1                  = 0x0000000000010000,   // Usage from external DLL analysis
    TetraTmo_Supress_UserDefined_2                  = 0x0000000000020000,   // Usage from external DLL analysis
    // ------------------------------------------------------------------------------------------------------------------------------------------------
    TetraDmo_Supress_Slot_1                         = 0x0000000000000001,   // Supress timeslot
    TetraDmo_Supress_Slot_2                         = 0x0000000000000002,   // Supress timeslot
    TetraDmo_Supress_Slot_3                         = 0x0000000000000004,   // Supress timeslot
    TetraDmo_Supress_Slot_4                         = 0x0000000000000008,   // Supress timeslot

    TetraDmo_Supress_NoCall_Slots                   = 0x0000000000000010,   // Supress timeslot
    TetraDmo_Supress_Master_Slots                   = 0x0000000000000020,   // Supress timeslot
    TetraDmo_Supress_Slave_Slots                    = 0x0000000000000040,   // Supress timeslot

    TetraDmo_Supress_NotDecodedBelowThreshhold      = 0x0000000000001000,   // Supress physical layer information
    TetraDmo_Supress_Repeated                       = 0x0000000000002000,   // Supress repeated messages(frame count down)
    TetraDmo_Supress_LateEntry                      = 0x0000000000004000,   // Supress DM-OCCPIED and DM-RESERVERD
    TetraDmo_Supress_USignal                        = 0x0000000000008000,   // Supress MAC-U-SIGNAL
    TetraDmo_Supress_Idle_DPresSync                 = 0x0000000000010000,   // Supress DPres sync with IDLE channel state
    TetraDmo_Supress_ErrorInPdu                     = 0x0000000000020000,   // Supress PDUs if an error occured
    // ------------------------------------------------------------------------------------------------------------------------------------------------
    Dmr_Supress_Slot_1                              = 0x0000000000000001,   // Supress timeslot
    Dmr_Supress_Slot_2                              = 0x0000000000000002,   // Supress timeslot

    Dmr_Supress_DL_IDLE_Pdu                         = 0x0000000000000010,   // Suppress IDLE-PDU
    Dmr_Supress_DL_EqualBroadCast                   = 0x0000000000000020,   // Suppress redunant broadcast
    Dmr_Supress_DL_NotDecoded                       = 0x0000000000000040,   // Supress physical layer information
    Dmr_Supress_DL_L2_CH_INFO                       = 0x0000000000000080,   // Supress SlotType/EMB/TACT if no higher layer present
    Dmr_Supress_DL_LateEntry                        = 0x0000000000000100,   // Supress late entry information in embedded channel
    Dmr_Supress_DL_ErrorInPdu                       = 0x0000000000100000,   // Supress PDUs if an error occured

    Dmr_Supress_UL_IDLE_Pdu                         = 0x0000000000001000,   // Suppress IDLE-PDU
    Dmr_Supress_UL_EqualBroadCast                   = 0x0000000000002000,   // Suppress redunant broadcast
    Dmr_Supress_UL_NotDecoded                       = 0x0000000000004000,   // Supress physical layer information, if they are below RSSI treshhold
    Dmr_Supress_UL_L2_CH_INFO                       = 0x0000000000008000,   // Supress SlotType/EMB/TACT if no higher layer present
    Dmr_Supress_UL_LateEntry                        = 0x0000000000010000,   // Supress late entry information in embedded channel
    Dmr_Supress_UL_ErrorInPdu                       = 0x0000000000200000    // Supress PDUs if an error occured
    // ------------------------------------------------------------------------------------------------------------------------------------------------
} EnumResultFilterSettings;
// ------------------------------------------------------------------------------------------------------

// ------------------------------------------------------------------------------------------------------
// Pointer for message data (use #undef this and renew #define for changeing)
#define __MSGDATA_POINTER                               pstMessageData

// Combine text generating and adding to MultiAnalyser
#define __Mas_PduOpen_sprintf( sLayer, sFormat, ... )   if ( (__MSGDATA_POINTER) && (__MSGDATA_POINTER)->callback_PduOpen )\
                                                        {   __Sec_Empty( g_strTextBuffer );\
                                                            __Sec_Add_sprintf( g_strTextBuffer, __Sec_sizeof(g_strTextBuffer), sFormat, __VA_ARGS__ );\
                                                            (__MSGDATA_POINTER)->callback_PduOpen( g_strTextBuffer, sLayer, (__MSGDATA_POINTER)->pInternalParameter );\
                                                        } else assert( !"__Mas_PduOpen_sprintf: No callback!" )

#define __Mas_PduText_sprintf( sFormat, ... )           if ( (__MSGDATA_POINTER) && (__MSGDATA_POINTER)->callback_PduAddText )\
                                                        {   __Sec_Empty( g_strTextBuffer );\
                                                            __Sec_Add_sprintf( g_strTextBuffer, __Sec_sizeof(g_strTextBuffer), sFormat, __VA_ARGS__ );\
                                                            (__MSGDATA_POINTER)->callback_PduAddText( g_strTextBuffer, (__MSGDATA_POINTER)->pInternalParameter );\
                                                        } else assert( !"__Mas_PduText_sprintf: No callback!" )

#define __Mas_PduText_strcpy( sString )                 if ( (__MSGDATA_POINTER) && (__MSGDATA_POINTER)->callback_PduAddText )\
                                                        {   (__MSGDATA_POINTER)->callback_PduAddText( sString, (__MSGDATA_POINTER)->pInternalParameter );\
                                                        } else assert( !"__Mas_PduText_strcpy: No callback!" )

#define __Mas_PduText_RawLine()                         __Mas_PduText_RawLine_Function( (__MSGDATA_POINTER)->ucData, (__MSGDATA_POINTER)->ulDataLength, (__MSGDATA_POINTER) );

#define __Mas_PduText_HexBinAsc( pData, nBytes )        __Mas_PduText_HexBinAsc_Function( (const unsigned char * const)pData, (const unsigned int)nBytes, __MSGDATA_POINTER )

#define __Mas_PduCose()                                 if ( (__MSGDATA_POINTER) && (__MSGDATA_POINTER)->callback_PduClose )\
                                                        {    (__MSGDATA_POINTER)->callback_PduClose( (__MSGDATA_POINTER)->pInternalParameter ); }\
                                                        else assert( !"__Mas_PduCose: No callback!" )
// ------------------------------------------------------------------------------------------------------


// ------------------------------------------------------------------------------------------------------
// DLL instances
#define DLL_INSTANCE_MAGIC      (0x1302643936721214)

// Instance data
typedef struct StructDllInstance
{
    // Pointer error detction
    unsigned long long   ullMagic;  // DLL_INSTANCE_MAGIC

    // Multi Instance support by chain list
    StructDllInstance   *pstNext;

    bool                 bNoExport;         // Disabled export
    unsigned char        ucPcapLinkType;    // 0=PPP, 1=Ethernet-Dummy header, 2=RAW-IP
    unsigned char        ucRfc1144;         // Try to decompress van jacobson IP/TCP header

    // Open PCAP FILE
    wchar_t              strMafFileName[_MAX_PATH];
    FILE                *pFileOpen;

    // van jacobsen tcp compression
    CRFC1144            *pclRfc144;
    unsigned char        ucIpBuffer[sizeof(StructThirdPartyComponent5V_Data::ucData)+256];

} StructDllInstance;
StructDllInstance   *g_pstDllInstance = NULL;

// Clean up DLL instance
static int Raw_Delete_DllInstance( StructDllInstance    *pstDll )
{
    StructDllInstance   *pstDllTemp = g_pstDllInstance, *pstLast = NULL;

    // Find DLL instance
    if ( NULL == pstDll || (DLL_INSTANCE_MAGIC != pstDll->ullMagic) )
    {   return -1;  }

    // Find instance in chain list
    while ( pstDllTemp )
    {
        assert( DLL_INSTANCE_MAGIC == pstDllTemp->ullMagic );
        // Found?
        if ( pstDllTemp == pstDll )
        { // Remove from chain list
            if ( pstLast )  {   pstLast->pstNext = pstDll->pstNext; }
            else            {   g_pstDllInstance = pstDll->pstNext; }
            break;
        }
        // Check next
        pstLast     = pstDllTemp;
        pstDllTemp  = pstDllTemp->pstNext;
    } // while ( pstDllTemp )

    // Close file
    if ( pstDll->pFileOpen )
    {   fclose( pstDll->pFileOpen ); pstDll->pFileOpen = NULL;  }

    if ( pstDll->pclRfc144 )
    {   delete pstDll->pclRfc144; pstDll->pclRfc144 = NULL; }

    memset( pstDll, 0, sizeof(pstDll[0]) );
    delete pstDll;

    return 0;
}

// Delete all instances
int Raw_Delete_All_DllInstances()
{
    // Find instance in chain list
    while ( g_pstDllInstance )
    {
        if ( Raw_Delete_DllInstance(g_pstDllInstance) )
        {   return -1;  }
    } // while ( g_pstDllInstance )

    return 0;
}

// Parse path+file+suffix into subparts: Filename + Suffix
bool Raw_GetPathFileNameNoSuffix( const wchar_t *pstrPathFile, wchar_t *pstrPathBasicFile, const unsigned int ulMaxPathLength )
{
    // Find last ".", "/" or "\"
    const intptr_t lPositionSlash   = max( __Sec_strrchr(pstrPathFile, L'/'), __Sec_strrchr(pstrPathFile, L'\\') );
    const intptr_t lPositionSuffix  = __Sec_strrchr(pstrPathFile, L'.');

    // Have suffix?
    if ( 0 > lPositionSuffix || (lPositionSlash > lPositionSuffix) )
    {   __Sec_strncpy2( pstrPathBasicFile, pstrPathFile, ulMaxPathLength );                                             }
    else
    {   __Sec_strncpy2( pstrPathBasicFile, pstrPathFile, 1+min((size_t)lPositionSuffix, (size_t)ulMaxPathLength-1) );   }

    return (0 != pstrPathBasicFile[0]);
}

// Parse path+file+suffix into subparts: Path
bool Raw_GetPathName( const wchar_t *pstrPathFile, wchar_t *pstrPath, const unsigned int ulMaxPathLength )
{
    // Find last "/" or "\"
    const intptr_t lPosition    = max( __Sec_strrchr(pstrPathFile, '/'), __Sec_strrchr(pstrPathFile, '\\') );

    // Clean string
    __Sec_Empty( pstrPath );

    // Have path?
    if ( (0 > lPosition) )
    {   return  false;  }

    // Copy path
    __Sec_strncpy2( pstrPath, pstrPathFile, min((size_t)(lPosition+2),(size_t)ulMaxPathLength) );

    return (0 != pstrPath[0]);
}
// ------------------------------------------------------------------------------------------------------

// Record (Packet) Header
typedef struct pcaprec_hdr_s
{
    unsigned long ts_sec;         // timestamp seconds
    unsigned long ts_usec;        // timestamp microseconds
    unsigned long incl_len;       // number of octets of packet saved in file
    unsigned long orig_len;       // actual length of packet
} pcaprec_hdr_t;

// Ethernet header
struct StructEtherHeader
{
    unsigned char   ucDestinationMACaddress[6];
    unsigned char   ucSourceMACaddress[6];
    unsigned char   ucEthernetType[2];
};

struct StructPPPHeader
{ // No flag and CRC version
    unsigned char   ucDirection;    // Extension by: LINKTYPE_PPP_WITH_DIR
    unsigned char   ucAddress;      // 0xFF
    unsigned char   ucControl;      // 0x03
    unsigned char   ucProtocol[2];
};

//struct StructPPP_Flag_Header
//{ // Start and End Flag plus CRC version
//  unsigned char   ucPPPFlag;
//  unsigned char   ucAddress;
//  unsigned char   ucControl;
//  unsigned char   ucProtocol[2];
//};
//struct StructPPP_Flag_End
//{ // Start and End Flag plus CRC version
//  unsigned char   ucCrc[2];
//  unsigned char   ucFlag;
//};

/*
 * FCS lookup table as calculated by the table generator in section B.2
 */
static const unsigned short fcstab[256] =\
{
   0x0000, 0x1189, 0x2312, 0x329b, 0x4624, 0x57ad, 0x6536, 0x74bf,
   0x8c48, 0x9dc1, 0xaf5a, 0xbed3, 0xca6c, 0xdbe5, 0xe97e, 0xf8f7,
   0x1081, 0x0108, 0x3393, 0x221a, 0x56a5, 0x472c, 0x75b7, 0x643e,
   0x9cc9, 0x8d40, 0xbfdb, 0xae52, 0xdaed, 0xcb64, 0xf9ff, 0xe876,
   0x2102, 0x308b, 0x0210, 0x1399, 0x6726, 0x76af, 0x4434, 0x55bd,
   0xad4a, 0xbcc3, 0x8e58, 0x9fd1, 0xeb6e, 0xfae7, 0xc87c, 0xd9f5,
   0x3183, 0x200a, 0x1291, 0x0318, 0x77a7, 0x662e, 0x54b5, 0x453c,
   0xbdcb, 0xac42, 0x9ed9, 0x8f50, 0xfbef, 0xea66, 0xd8fd, 0xc974,
   0x4204, 0x538d, 0x6116, 0x709f, 0x0420, 0x15a9, 0x2732, 0x36bb,
   0xce4c, 0xdfc5, 0xed5e, 0xfcd7, 0x8868, 0x99e1, 0xab7a, 0xbaf3,
   0x5285, 0x430c, 0x7197, 0x601e, 0x14a1, 0x0528, 0x37b3, 0x263a,
   0xdecd, 0xcf44, 0xfddf, 0xec56, 0x98e9, 0x8960, 0xbbfb, 0xaa72,
   0x6306, 0x728f, 0x4014, 0x519d, 0x2522, 0x34ab, 0x0630, 0x17b9,
   0xef4e, 0xfec7, 0xcc5c, 0xddd5, 0xa96a, 0xb8e3, 0x8a78, 0x9bf1,
   0x7387, 0x620e, 0x5095, 0x411c, 0x35a3, 0x242a, 0x16b1, 0x0738,
   0xffcf, 0xee46, 0xdcdd, 0xcd54, 0xb9eb, 0xa862, 0x9af9, 0x8b70,
   0x8408, 0x9581, 0xa71a, 0xb693, 0xc22c, 0xd3a5, 0xe13e, 0xf0b7,
   0x0840, 0x19c9, 0x2b52, 0x3adb, 0x4e64, 0x5fed, 0x6d76, 0x7cff,
   0x9489, 0x8500, 0xb79b, 0xa612, 0xd2ad, 0xc324, 0xf1bf, 0xe036,
   0x18c1, 0x0948, 0x3bd3, 0x2a5a, 0x5ee5, 0x4f6c, 0x7df7, 0x6c7e,
   0xa50a, 0xb483, 0x8618, 0x9791, 0xe32e, 0xf2a7, 0xc03c, 0xd1b5,
   0x2942, 0x38cb, 0x0a50, 0x1bd9, 0x6f66, 0x7eef, 0x4c74, 0x5dfd,
   0xb58b, 0xa402, 0x9699, 0x8710, 0xf3af, 0xe226, 0xd0bd, 0xc134,
   0x39c3, 0x284a, 0x1ad1, 0x0b58, 0x7fe7, 0x6e6e, 0x5cf5, 0x4d7c,
   0xc60c, 0xd785, 0xe51e, 0xf497, 0x8028, 0x91a1, 0xa33a, 0xb2b3,
   0x4a44, 0x5bcd, 0x6956, 0x78df, 0x0c60, 0x1de9, 0x2f72, 0x3efb,
   0xd68d, 0xc704, 0xf59f, 0xe416, 0x90a9, 0x8120, 0xb3bb, 0xa232,
   0x5ac5, 0x4b4c, 0x79d7, 0x685e, 0x1ce1, 0x0d68, 0x3ff3, 0x2e7a,
   0xe70e, 0xf687, 0xc41c, 0xd595, 0xa12a, 0xb0a3, 0x8238, 0x93b1,
   0x6b46, 0x7acf, 0x4854, 0x59dd, 0x2d62, 0x3ceb, 0x0e70, 0x1ff9,
   0xf78f, 0xe606, 0xd49d, 0xc514, 0xb1ab, 0xa022, 0x92b9, 0x8330,
   0x7bc7, 0x6a4e, 0x58d5, 0x495c, 0x3de3, 0x2c6a, 0x1ef1, 0x0f78
   };

#define PPPINITFCS16    0xffff  /* Initial FCS value */
#define PPPGOODFCS16    0xf0b8  /* Good final FCS value */

/*
* Calculate a new fcs given the current fcs and the new data.
 */
unsigned short pppfcs16(unsigned short fcs, const unsigned char *cp, int len)
{
    while ( len-- )
    {   fcs = (fcs >> 8) ^ fcstab[(fcs ^ *cp++) & 0xff];    }

    return (fcs);
}

///*
// * How to use the fcs
// */
//tryfcs16(cp, len)
//    register unsigned char *cp;
//    register int len;
//{
//    u16 trialfcs;
//
//    /* add on output */
//    trialfcs = pppfcs16( PPPINITFCS16, cp, len );
//    trialfcs ^= 0xffff;             /* complement */
//    cp[len] = (trialfcs & 0x00ff);  /* least significant byte first */
//    cp[len+1] = ((trialfcs >> 8) & 0x00ff);
//
//    /* check on input */
//    trialfcs = pppfcs16( PPPINITFCS16, cp, len + 2 );
//    if ( trialfcs == PPPGOODFCS16 )
//        printf("Good FCS0);
//}

// Read configuration for export
void Raw_Open_Configuration( StructDllInstance  *pstDll )
{
    wchar_t strFileName[_MAX_PATH] = { 0 };
    FILE *pFileOpen = NULL;

    // Make default
    pstDll->bNoExport       = true; // Must be enabled
    pstDll->ucPcapLinkType  = 0;    // PPP
    pstDll->ucRfc1144       = 0;    // No van Jacobson decompression

    // Open configuration file in the path
    if ( Raw_GetPathName(pstDll->strMafFileName, strFileName, _MAX_PATH) && strFileName[0] )
    { // Add filename and open
        __Sec_Add_sprintf( strFileName, _MAX_PATH, L"\\MultiAnalyzerIpExport.txt" );
        pFileOpen = _wfsopen( strFileName, L"rt", _SH_DENYNO );
    } // if ( Raw_GetPathName(pstDll->strMafFileName, strFileName, _MAX_PATH) && strFileName[0] )
    // Not open try on desktop
    if ( NULL == pFileOpen )
    { // Get Desktop
        if ( SHGetSpecialFolderPath(HWND_DESKTOP, strFileName, CSIDL_DESKTOP, FALSE) && strFileName[0] )
        { // Add filename and open
            __Sec_Add_sprintf( strFileName, _MAX_PATH, L"\\MultiAnalyzerIpExport.txt" );
            pFileOpen = _wfsopen( strFileName, L"rt", _SH_DENYNO );
        }
    } // if ( NULL == pFileOpen )
    // Not open try some other pathes
    if ( NULL == pFileOpen )
    {
        const wchar_t *pstrCfgDirs[] =
        {
            L"USERPROFILE",
            L"APPDATA",
            L"LOCALAPPDATA",
            L"ProgramFiles" // Special one (install drawer)
        };
        // Try all
        for ( unsigned int ulKeyWord = 0; ARRAYSIZE(pstrCfgDirs) > ulKeyWord; ulKeyWord++ )
        { // Get more possiblities
            strFileName[0] = 0; wchar_t *pstrPath = strFileName; size_t tBufferLength = _MAX_PATH;
            if ( _wdupenv_s( &pstrPath, &tBufferLength, pstrCfgDirs[ulKeyWord] ) || (nullptr == pstrPath) || (0 == pstrPath[0]) )
            {   continue;   }
            // Add filename and open
            if ( (ARRAYSIZE(pstrCfgDirs)-1) == ulKeyWord )
            {   __Sec_Add_sprintf( strFileName, _MAX_PATH, L"%s\\MultiAnalyzer\\MultiAnalyzerIpExport.txt", pstrPath ); }
            else
            {   __Sec_Add_sprintf( strFileName, _MAX_PATH, L"%s\\MultiAnalyzerIpExport.txt", pstrPath );    }
            pFileOpen = _wfsopen( strFileName, L"rt", _SH_DENYNO );
            if ( pFileOpen )
            {   break;  }
        } // for ( unsigned int ulKeyWord = 0; ARRAYSIZE(pstrCfgDirs) > ulKeyWord; ulKeyWord++ )
    } // if ( NULL == pFileOpen )

    // Read file if open
    if ( pFileOpen )
    {
        wchar_t strLine[80] = { 0 };
        const struct StructCfgKey
        {
            const wchar_t   *pstrKey;
            const size_t     tKeySize;
        } stKeys[] =
        {
            {   L"EXPORT="      , wcsnlen(L"EXPORT="  ,80)    },  // Argumnet: TRUE, FALSE
            {   L"LINKTYPE="    , wcsnlen(L"LINKTYPE=",80)    },  // Argumnet: PPP, ETHERNET, IPRAW
            {   L"RFC1144="     , wcsnlen(L"RFC1144=" ,80)    }   // Argumnet: TRUE, FALSE
        };

        // Read configurion
        while( 1 == fwscanf_s( pFileOpen, L"%79s\n", strLine, (unsigned int)_countof(strLine) ) )
        {
            strLine[ARRAYSIZE(strLine)-1] = 0;
            // Check for keywords
            for ( unsigned int ulKeyWord = 0; ARRAYSIZE(stKeys) > ulKeyWord; ulKeyWord++ )
            {
                if ( 0 != _wcsnicmp(strLine, stKeys[ulKeyWord].pstrKey, stKeys[ulKeyWord].tKeySize) )
                {   continue;   }

                switch ( ulKeyWord )
                {
                    case 0: // "EXPORT="; Argumnet: ON, OFF (TRUE, FALSE)
                        if ( 0 == _wcsnicmp(&strLine[stKeys[ulKeyWord].tKeySize], L"TRUE"    , 4) ) {   pstDll->bNoExport       = false;} else
                                                                                                    {   pstDll->bNoExport       = true; }
                    break;
                    case 1: // "FILETYPE="; Argumnet: PPP, ETHERNET, IPRAW
                        if ( 0 == _wcsnicmp(&strLine[stKeys[ulKeyWord].tKeySize], L"PPP"     , 3) ) {   pstDll->ucPcapLinkType  = 0;    } else
                        if ( 0 == _wcsnicmp(&strLine[stKeys[ulKeyWord].tKeySize], L"ETHERNET", 8) ) {   pstDll->ucPcapLinkType  = 1;    } else
                        if ( 0 == _wcsnicmp(&strLine[stKeys[ulKeyWord].tKeySize], L"IPRAW"   , 5) ) {   pstDll->ucPcapLinkType  = 2;    }
                    break;
                    case 2: // "RFC1144="; Argumnet: TRUE, FALSE
                        if ( 0 == _wcsnicmp(&strLine[stKeys[ulKeyWord].tKeySize], L"TRUE"    , 4) ) {   pstDll->ucRfc1144       = 1;    } else
                                                                                                    {   pstDll->ucRfc1144       = 0;    }
                    break;
                    default:
                        assert( !"ulKeyWord unsupported" );
                    break;
                } // switch ( ulKeyWord )
                // Key word handled
                break;
            } // for ( unsigned int ulKeyWord = 0; ARRAYSIZE(stKeys) > ulKeyWord; ulKeyWord++ )
        } // while( 1 == wscanf_s(L"%s\n"), strLine, ARRAYSIZE(strLine) )
        // Close file
        fclose(pFileOpen);
    } // if ( pFileOpen )

    // Need van Jacobson class?
    if ( (pstDll->ucRfc1144) && (NULL == pstDll->pclRfc144) )
    {    pstDll->pclRfc144  = new (std::nothrow)CRFC1144;   }
    else
    if ( (!pstDll->ucRfc1144) && (pstDll->pclRfc144) )
    {   delete pstDll->pclRfc144; pstDll->pclRfc144 = NULL; }
}

// Open PCAP file
bool Raw_Open_PCAP_FILE( StructDllInstance  *pstDll, const time_t tTime )
{
    wchar_t strFileName[_MAX_PATH] = { 0 };

    // Ensure closed file
    if ( pstDll->pFileOpen )    {   fclose( pstDll->pFileOpen ); pstDll->pFileOpen = NULL;  }

    // Load configuration
    Raw_Open_Configuration( pstDll );
    if ( pstDll->bNoExport )
    {   return false;   }

    // Open filename, have MAF file for drawer and basic filename?
    if ( false == Raw_GetPathFileNameNoSuffix(pstDll->strMafFileName, strFileName, _MAX_PATH) )
    { // Generate dummy on desktop
#if (defined WIN32)
        tm stTime; localtime_s( &stTime, &tTime );
        #define pTIME   (&stTime)
        if ( !SHGetSpecialFolderPath(HWND_DESKTOP, strFileName, CSIDL_DESKTOP, FALSE) )
        {   return NULL;    }
#else
        return NULL;
        //const tm *pstTm = localtime( pstTime->tTime );
        //#define pTIME pstTm
#endif // (defined WIN32)
        __Sec_Add_sprintf( strFileName, _MAX_PATH, L"\\%4.4u-%2.2u-%2.2u_%2.2u.%2.2u__MasExportIP", (1900+pTIME->tm_year), (1+pTIME->tm_mon), pTIME->tm_mday, pTIME->tm_hour, pTIME->tm_min );
    } // if ( false == Raw_GetPathFileNameNoSuffix(pstDll->strMafFileName, strFileName,_MAX_PATH) )

    // Add new suffix
    __Sec_Add_sprintf( strFileName, _MAX_PATH, L".pcap" );

    // Try to open file
    pstDll->pFileOpen = _wfsopen( strFileName, L"wb", _SH_DENYNO );
    if ( pstDll->pFileOpen )
    { // Write PCAP header
        typedef struct pcap_hdr_s
        {
            unsigned long  magic_number;    // magic number
            unsigned short version_major;   // major version number
            unsigned short version_minor;   // minor version number
            long           thiszone;        // GMT to local correction
            unsigned long  sigfigs;         // accuracy of timestamps
            unsigned long  snaplen;         // max length of captured packets, in octets
            unsigned long  network;         // data link type
        } pcap_hdr_t;
        #define LINKTYPE_ETHERNET       1           // DLT_EN10MB       IEEE 802.3 Ethernet (10Mb, 100Mb, 1000Mb, and up); the 10MB in the DLT_ name is historical.
        //#define LINKTYPE_ENCAP_PPP    9           // DLT_PPP          PPP, as per RFC 1661 and RFC 1662; if the first 2 bytes are 0xff and 0x03, it's PPP in HDLC-like framing, with the PPP header following those two bytes, otherwise it's PPP without framing, and the packet begins with the PPP header. The data in the frame is not octet-stuffed or bit-stuffed.
        #define LINKTYPE_PPP_WITH_DIR   204         // DLT_PPP_WITH_DIR PPP, as per RFC 1661 and RFC 1662, preceded with a one-byte pseudo-header with a zero value meaning "received by this host" and a non-zero value meaning "sent by this host"; if the first 2 bytes are 0xff and 0x03, it's PPP in HDLC-like framing, with the PPP header following those two bytes, otherwise it's PPP without framing, and the packet begins with the PPP header. The data in the frame is not octet-stuffed or bit-stuffed.
        #define LINKTYPE_RAW            101         // DLT_RAW          Raw IP; the packet begins with an IPv4 or IPv6 header, with the "version" field of the header indicating whether it's an IPv4 or IPv6 header.
        // Generate header
        pcap_hdr_t stHeader =
        {
            0xa1b2c3d4,         // magic number
            2,                  // major version number
            4,                  // minor version number
            0,                  // GMT to local correction
            0,                  // accuracy of timestamps
            65535,              // max length of captured packets, in octets
            LINKTYPE_RAW        // data link type
        };
        switch ( pstDll->ucPcapLinkType )
        {
            default:
                assert( !"ucPcapLinkType unsupported!" );
//          case 0: stHeader.snaplen = 65535+sizeof(pcaprec_hdr_t)+sizeof(StructPPPHeader);   stHeader.network = LINKTYPE_ENCAP_PPP;    break;  // PPP (NO start/end flags and CRC)
            case 0: stHeader.snaplen = 65535+sizeof(pcaprec_hdr_t)+sizeof(StructPPPHeader);   stHeader.network = LINKTYPE_PPP_WITH_DIR; break;  // PPP (NO start/end flags and CRC)
            case 1: stHeader.snaplen = 65535+sizeof(pcaprec_hdr_t)+sizeof(StructEtherHeader); stHeader.network = LINKTYPE_ETHERNET;     break;  // ETHERNET
            case 2: stHeader.snaplen = 65535+sizeof(pcaprec_hdr_t);                           stHeader.network = LINKTYPE_RAW;          break;  // IP RAW
        }
        // Write fileformart header
        if ( sizeof(stHeader) != fwrite( &stHeader, 1, sizeof(stHeader), pstDll->pFileOpen ) )
        {   fclose( pstDll->pFileOpen ); pstDll->pFileOpen = NULL;  }
    } // if ( pstDll->pFileOpen )

    return (NULL != pstDll->pFileOpen);
}

// DLL provides wanted types of data for it own analysing...
// Returns 0: If successfull, all other values marking an error
THIRD_PARTY_COMPONENT_5V int ThirdPartyComponent5V_Init( StructThirdPartyComponent5V_Init *pstRegisterMessages )
{
    StructThirdPartyComponent5V_RegData *pstRegData = NULL;    if ( NULL == pstRegisterMessages )
    {   return -1;  }

    // Valid result struct?
    switch ( pstRegisterMessages->ulVersion )
    {
        case 1: // First version of struct
        break;

        default:
        return -2;
    } //  switch ( pstRegisterMessages->ulVersion )

     // Version/Company/Copyright string
    __Sec_strcpy_asc( pstRegisterMessages->strVersion   , "Export AL IP data, Version 1.0" );

    // Supported Protokoll, which type of protokoll data is supported
    pstRegisterMessages->ulProtokollSupport     = ThirdPartyComponent5V_TetraTmo;
    pstRegisterMessages->pDllInstance           = NULL;

    // All registred data
    pstRegisterMessages->ulRegisterCount        = 0;
    memset( pstRegisterMessages->stRegisterData , 0 , sizeof(pstRegisterMessages->stRegisterData) );

    // --------------------
    // Set data: Analyse registred PDU* NOTE1                   ulType: SNDCP-PDU Type              , ulExtendedType: -
    pstRegData = &pstRegisterMessages->stRegisterData[ pstRegisterMessages->ulRegisterCount ];
    pstRegData->eRegisterType     = ThirdPartyComponent5V_Sndcp_Content;
    pstRegData->ulDirection       = 0x03;     // DL+UL
    pstRegData->ulType            = UINT_MAX; // ALL
    pstRegData->ulExtendedType[0] = UINT_MAX; // ALL
    pstRegData->ulExtendedType[1] = UINT_MAX; // ALL
    pstRegisterMessages->ulRegisterCount++;
    // --------------------

    // --------------------
    // Make new instance
    StructDllInstance   *pstDll = new (std::nothrow)StructDllInstance;
    if ( NULL == pstDll )
    {   return -1;  }
    memset( pstDll          , 0, sizeof(pstDll[0]) );
    pstDll->ullMagic        = DLL_INSTANCE_MAGIC;
    pstDll->pclRfc144       = NULL; // new (std::nothrow)CRFC1144;
    pstDll->bNoExport       = false;// Assume export ok
    pstDll->ucPcapLinkType  = 0;    // PPP
    pstDll->ucRfc1144       = 0;    // FALASE

    // Add into chain list
    pstDll->pstNext     = g_pstDllInstance;
    g_pstDllInstance    = pstDll;

    // Store instance
    pstRegisterMessages->pDllInstance   = pstDll;
    // --------------------

    // All done
    return 0;
}

// Multianalyser provides data for analysing:
// Input : StructThirdPartyComponent5V_Data  : The data themself for external analysing
// Output: StructThirdPartyComponent5V_Result: What is todo for multianalyser.
// Returns 0: Result is stored in StructThirdPartyComponent5V_Result, all other values are errors
THIRD_PARTY_COMPONENT_5V int ThirdPartyComponent5V_Data( void *pDllInstance, const StructThirdPartyComponent5V_Data *pstMessageData, StructThirdPartyComponent5V_Result *pstAnalyseResult )
{
    StructDllInstance   *pstDll         = (StructDllInstance*)pDllInstance;

    // Correct instance?
    if ( (NULL == pstDll) || (DLL_INSTANCE_MAGIC != pstDll->ullMagic) )
    {   return -5;  }

    // Have result struct?
    if ( NULL == (__MSGDATA_POINTER))
    {   return -1;  }

    // Have result struct?
    if ( NULL == pstAnalyseResult)
    {   return -2;  }

    // Valid result struct?
    switch ( (__MSGDATA_POINTER)->ulVersion )
    { // Known versions
        case STRUCT_THIRDPARTYCOMPONENT5V_DATA__VERSION_V1:
        break;
        case STRUCT_THIRDPARTYCOMPONENT5V_DATA__VERSION_V2:
        case STRUCT_THIRDPARTYCOMPONENT5V_DATA__VERSION_V3:
        // Information about MAF file
            if ( (((const StructThirdPartyComponent5V_DataV2*)(__MSGDATA_POINTER))->bNewMafFile) || (0 == pstDll->strMafFileName[0]) )
            { // Store file name
                __Sec_strcpy(  pstDll->strMafFileName, ((const StructThirdPartyComponent5V_DataV2*)(__MSGDATA_POINTER))->pstrCurrentMafFileName );
                // Close old to ensure reopen new one...
                if ( pstDll->pFileOpen )
                {   fclose(pstDll->pFileOpen); pstDll->pFileOpen = NULL;    }
            }
        break;
        default:
        return -3;
    } // switch (  (__MSGDATA_POINTER)->ulVersion )

    // Valid data?
    if ( STRUCT_THIRDPARTYCOMPONENT5V_RESULT__VERSION != pstAnalyseResult->ulVersion )
    {   return -4;  }

    // Ensure nothing returned
    pstAnalyseResult->eResult           = ThirdPartyComponent5V_Result_Nothing;
    pstAnalyseResult->ullFilterMask     = EnumResultFilterSettings::Filter_ViewAll;
    pstAnalyseResult->ulAnalyseLength   = 0;

    // Depending from message type write text for MSC
    switch ((__MSGDATA_POINTER)->eRegisterType )
    {
        case ThirdPartyComponent5V_Sndcp_Content:
        { // Acceptable length
            unsigned int         ulHeaderCompression= pstMessageData->ulExtendedType[0]; // PCOMP for compression
            unsigned int         ulDataCompression  = pstMessageData->ulExtendedType[1]; // DCOMP for compression
            unsigned long        ulDataLengthInByte = (pstMessageData->ulDataLength >> 3);
            const unsigned char *pucData            = pstMessageData->ucData;
            assert( 0 == (7 & pstMessageData->ulDataLength) );

            // Want to export something?
            if ( pstDll->bNoExport )
            {   break;  }

            //if ( (20 > ulDataLengthInByte) || (65535 < ulDataLengthInByte) )
            //{ assert( (20 <= ulDataLengthInByte) && (65535 >= ulDataLengthInByte) );      break;  }
            // Need to open PCAP file?
            if ( NULL == pstDll->pFileOpen ) {  if ( !Raw_Open_PCAP_FILE(pstDll, pstMessageData->ullTime) ) {   break;  } }

            // Is data compressed "DCOMP"
            switch ( ulDataCompression )
            {
                case 0: // No compression
                break;

                default:
                //case 1: // Recommendation ITU-T V.42bis [27]
                //case 2: // BSD compression
                //case 3: // Predictor compression
                //case 4: // BSD uncompressible packet
                    assert( !"TODO: data compression" );
                break;
            } // switch ( ulDataCompression )
            // Can write data?
            if ( ulDataCompression )
            {   break;  }

            // Want to decompress van Jacobson IP/TCP header
            if ( pstDll->pclRfc144 )
            switch ( ulHeaderCompression )
            {
                case 1: // Van Jacobson compressed TCP/IP
                if ( (sizeof(pstDll->ucIpBuffer)-256) >= ulDataLengthInByte )
                {
                    memcpy( pstDll->ucIpBuffer, pstMessageData->ucData, ulDataLengthInByte );
                    // Decompress
                    pucData = pstDll->pclRfc144->Decompress( pstDll->ucIpBuffer, ulDataLengthInByte, (1 == pstMessageData->ulDirection), true );
                    // Decompressed?
                    if ( NULL != pucData )
                    { // If uncompressed, set new length from IP header
                        const unsigned int ulNewIpLengthInByte = (((unsigned long)pucData[2]) << 8) | (((unsigned long)pucData[3]));
                        // Possible size?
                        if ( (sizeof(pstDll->ucIpBuffer) >= ulNewIpLengthInByte) && (ulNewIpLengthInByte > ulDataLengthInByte) &&
                            (20                          <= ulNewIpLengthInByte) && ((ulDataLengthInByte+128) >= ulNewIpLengthInByte) )
                        {   ulHeaderCompression = 0; ulDataLengthInByte = ulNewIpLengthInByte; break;   } // Ok, so use decompressed data
                    }
                    // Failed to decompress, use orginal data
                    pucData = pstMessageData->ucData; // Set back to RAW
                }
                break;
                case 2: // Van Jacobson non-compressed TCP/IP
                    (void)pstDll->pclRfc144->Decompress( (unsigned char*)pstMessageData->ucData, ulDataLengthInByte, (1 == pstMessageData->ulDirection), false );
                    ulHeaderCompression = 0;
                break;
                default:
                    // Nothing todo...
                break;
            } // switch ( ulHeaderCompression )

            // Depending from file type write data
            switch ( pstDll->ucPcapLinkType )
            {
                default:
                    assert( !"ucPcapLinkType unsupported!" );
                case 0: // PPP (NO start/end flags and CRC)
                { // Make PCAP single message header
                    const pcaprec_hdr_t stPacket =
                    {
                              (unsigned long)pstMessageData->ullTime,           // timestamp seconds
                        (1000*(unsigned long)pstMessageData->ulMilliSeconds),   // timestamp microseconds
                        sizeof(StructPPPHeader)+(unsigned long)ulDataLengthInByte,  // number of octets of packet saved in file
                        sizeof(StructPPPHeader)+(unsigned long)ulDataLengthInByte,  // actual length of packet
                    };
                    // Write header data
                    if ( sizeof(stPacket) != fwrite( &stPacket, 1, sizeof(stPacket), pstDll->pFileOpen ) )
                    {   assert( !"Can not write ulDataLength" ); break;     }
                    struct StructPPPHeader stPPPHeader =
                    {
                        (1 != pstMessageData->ulDirection), // ucDirection; Extension by: LINKTYPE_PPP_WITH_DIR
                        //0x7E,                             // ucPPPFlag; (for StructPPP_Flag_Header)
                        0xFF,                               // ucAddress;
                        0x03,                               // ucControl;
                        {0,0x21}                            // IP, Internet Protocol, see below:
                    };
                    // Check PCOMP for compression
                    switch ( ulHeaderCompression )
                    {
                        case 0:  stPPPHeader.ucProtocol[0] = 0x00; stPPPHeader.ucProtocol[1] = 0x21;    break; // No compression                    : 0x0021    IP, Internet Protocol.
                        case 1:  stPPPHeader.ucProtocol[0] = 0x00; stPPPHeader.ucProtocol[1] = 0x2D;    break; // Van Jacobson compressed TCP/IP    : 0x002D    Van Jacobson Compressed TCP/IP.
                        case 2:  stPPPHeader.ucProtocol[0] = 0x00; stPPPHeader.ucProtocol[1] = 0x2F;    break; // Van Jacobson non-compressed TCP/IP: 0x002F    Van Jacobson Uncompressed TCP/IP.
                        case 3:  stPPPHeader.ucProtocol[0] = 0x00; stPPPHeader.ucProtocol[1] = 0x61;    break; // FULL_HEADER                       : 0x0061    RTP IPHC Full Header.
                        case 4:  stPPPHeader.ucProtocol[0] = 0x00; stPPPHeader.ucProtocol[1] = 0x63;    break; // COMPRESSED_TCP                    : 0x0063    RTP IPHC Compressed TCP.
                        case 5:  stPPPHeader.ucProtocol[0] = 0x20; stPPPHeader.ucProtocol[1] = 0x63;    break; // COMPRESSED_TCP_NODELTA            : 0x2063    RTP IPHC Compressed TCP No Delta.
                        case 6:  stPPPHeader.ucProtocol[0] = 0x00; stPPPHeader.ucProtocol[1] = 0x65;    break; // COMPRESSED_NON_TCP                : 0x0065    RTP IPHC Compressed Non TCP.
                        case 7:  stPPPHeader.ucProtocol[0] = 0x00; stPPPHeader.ucProtocol[1] = 0x69;    break; // COMPRESSED_RTP_8                  : 0x0069    RTP IPHC Compressed RTP 8.
                        case 8:  stPPPHeader.ucProtocol[0] = 0x20; stPPPHeader.ucProtocol[1] = 0x69;    break; // COMPRESSED_RTP_16                 : 0x2069    RTP IPHC Compressed RTP 16.
                        case 9:  stPPPHeader.ucProtocol[0] = 0x00; stPPPHeader.ucProtocol[1] = 0x67;    break; // COMPRESSED_UDP_8                  : 0x0067    RTP IPHC Compressed UDP 8.
                        case 10: stPPPHeader.ucProtocol[0] = 0x20; stPPPHeader.ucProtocol[1] = 0x67;    break; // COMPRESSED_UDP_16                 : 0x2067    RTP IPHC Compressed UDP 16.
                        case 11: stPPPHeader.ucProtocol[0] = 0x20; stPPPHeader.ucProtocol[1] = 0x65;    break; // CONTEXT_STATE                     : 0x2065    RTP IPHC Context State.
                    } // switch ( ulHeaderCompression )
                    // Make header
                    if ( sizeof(stPPPHeader) != fwrite( &stPPPHeader, 1, sizeof(stPPPHeader), pstDll->pFileOpen ) )
                    {   assert( !"Can not write ulDataLength" ); break;     }
                }
                break;
                case 1: // ETHERNET
                { // No header compression support for file type
                    if ( ulHeaderCompression )
                    {   break;  }

                    // Make PCAP single message header
                    const pcaprec_hdr_t stPacket =
                    {
                              (unsigned long)pstMessageData->ullTime,           // timestamp seconds
                        (1000*(unsigned long)pstMessageData->ulMilliSeconds),   // timestamp microseconds
                           14+(unsigned long)ulDataLengthInByte,                // number of octets of packet saved in file
                           14+(unsigned long)ulDataLengthInByte,                // actual length of packet
                    };
                    // Write header data
                    if ( sizeof(stPacket) != fwrite( &stPacket, 1, sizeof(stPacket), pstDll->pFileOpen ) )
                    {   assert( !"Can not write ulDataLength" ); break;     }

                    // Writer dummy ethernet header (use as MAC address source and target TETRA address)
                    const StructEtherHeader stEtherHeader =
                    {
                        {   0, 0, 0, (0xFF & (pstMessageData->ulTerminatorAddress >> 16)), (0xFF & (pstMessageData->ulTerminatorAddress >> 8)), (0xFF & (pstMessageData->ulTerminatorAddress)) },
                        {   0, 0, 0, (0xFF & (pstMessageData->ulOrginatorAddress  >> 16)), (0xFF & (pstMessageData->ulOrginatorAddress  >> 8)), (0xFF & (pstMessageData->ulOrginatorAddress))  },
                        {   0x08, 0x00  }   // IPv4
                    };
                    // Write ethernet header data
                    if ( sizeof(stEtherHeader) != fwrite( &stEtherHeader, 1, sizeof(stEtherHeader), pstDll->pFileOpen ) )
                    {   assert( !"Can not write ulDataLength" ); break;     }
                }
                break;
                case 2: // IP RAW
                { // No header compression support for file type
                    if ( ulHeaderCompression )
                    {   break;  }

                    // Make PCAP single message header
                    const pcaprec_hdr_t stPacket =
                    {
                              (unsigned long)pstMessageData->ullTime,           // timestamp seconds
                        (1000*(unsigned long)pstMessageData->ulMilliSeconds),   // timestamp microseconds
                              (unsigned long)ulDataLengthInByte,                // number of octets of packet saved in file
                              (unsigned long)ulDataLengthInByte,                // actual length of packet
                    };
                    // Write header data
                    if ( sizeof(stPacket) != fwrite( &stPacket, 1, sizeof(stPacket), pstDll->pFileOpen ) )
                    {   assert( !"Can not write ulDataLength" ); break;     }
                }
                break;
            } // switch ( pstDll->ucPcapLinkType )

            // Write Payload
            if ( ulDataLengthInByte != fwrite(pucData, 1, ulDataLengthInByte, pstDll->pFileOpen) )
            {   assert( !"Can not write ulDataLength" ); break;     }

            // Not needed for supprorted file formarts
            //switch ( pstDll->ucPcapLinkType )
            //{
            //  case 0: // PPP (with CRC and flag)
            //  { // Terminate PPP message
            //      const unsigned short usCrc = pppfcs16(PPPINITFCS16, pstMessageData->ucData,  ulDataLengthInByte);
            //      const struct StructPPP_Flag_End stPPPEnd =
            //      {
            //          {(0xFF & (usCrc >> 8)),(0xFF & (usCrc))},   // ucCrc[2]
            //          0x7E                                        // ucFlag;
            //      };
            //      if ( sizeof(stPPPEnd) != fwrite( &stPPPEnd, 1, sizeof(stPPPEnd), pstDll->pFileOpen ) )
            //      {   assert( !"Can not write ulDataLength" ); break;     }
            //  } // PPP (with CRC and flag)
            //  break;
            //  default:
            //  break;
            //} // switch ( pstDll->ucPcapLinkType )
        } // case ThirdPartyComponent5V_Sndcp_Content:
        break;

        default:
        break;
    } // switch ((__MSGDATA_POINTER)->eRegisterType )

    // Result in "pstAnalyseResult" *NOT* stored (so MA does not need to care)
    return -1;
}

// Called if no further data will be analysed
THIRD_PARTY_COMPONENT_5V int ThirdPartyComponent5V_Close(void *pDllInstance)
{
    StructDllInstance   *pstDll = (StructDllInstance*)pDllInstance, *pstDllTemp = g_pstDllInstance, *pstLast = NULL;

    // Delete instance
    (int)Raw_Delete_DllInstance( (StructDllInstance*)pDllInstance );

    // Nothing todo yet
    return 0;
}
