/*
 * Zero-Clause BSD
 *
 * Permission to use, copy, modify, and/or distribute this software for any purpose with or without fee is hereby granted.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY SPECIAL, DIRECT, INDIRECT, OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 * NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

//#undef _WINDOWS
//#undef WIN32

#if ((defined _WINDOWS) || (defined WIN32) || (defined WIN64))
    #include <Windows.h>    // Standard windows includes
#endif // ((defined _WINDOWS) || (defined WIN32) || (defined WIN64))
#include "..\ThirdPartyComponent5V.h"   // Me self...
#include <wchar.h>          // wcsnlen_s(), wcsncpy_s(), swprintf_s()
#include <string.h>         // strncpy_s()
#include <assert.h>         // assert()
#include <shlobj.h>         // SHGetSpecialFolderPath
#include <new>              // new (std::nothrow)
#include <cstdio>           // FILE

// ------------------------------------------------------------------------------------------------------
#if !((defined _WINDOWS) || (defined WIN32) || (defined WIN64))
    // Define POSIX versions
    #define wcsnlen_s           wcsnlen
    #define wcsncpy_s           wcsncpy
    #define strncpy_s           strncpy
    #define swprintf_s          swprintf
#endif // !((defined _WINDOWS) || (defined WIN32) || (defined WIN64))

// Macros and function for secure add text to a buffer
#ifndef min
    #define __Sec_Min( a, b )                                       (((a) < (b)) ? (a) : (b))
#else  // min
    #define __Sec_Min                                               min
#endif // min
#ifndef max
    #define __Sec_Max( a, b )                                       (((a) > (b)) ? (a) : (b))
#else  // min
    #define __Sec_Max                                               max
#endif // min

// Clean string
#define __Sec_Empty( sDestString )                                  (sDestString)[0] = 0

// Size of the string
#define __Sec_sizeof( sDestString )                                 ((size_t)sizeof(sDestString)/(size_t)sizeof((sDestString)[0]))
#define __Sec_sizeofN( sDestString, nMaxLength )                    __Sec_Min(__Sec_sizeof(sDestString), nMaxLength)

// Force to terminate string
#define __Sec_CloseN( sDestString, nMaxLength )                     sDestString[__Sec_Max(__Sec_sizeofN(sDestString,nMaxLength),1)-1] = 0;

// Give pack string length
#define __Sec_Strnlen( sDestString, nMaxLength )                    wcsnlen_s( sDestString, __Sec_sizeofN(sDestString, nMaxLength) )

// Give back remaining string length
#define __Sec_Remaining_strlen( sDestString, nMaxLength )           (__Sec_sizeofN(sDestString, nMaxLength) - __Sec_Strnlen(sDestString, nMaxLength))

// Add new formarted string
#define __Sec_Add_sprintf( sDestString, nMaxLength, sFormat, ... )  if ( (1 < __Sec_Remaining_strlen(sDestString, nMaxLength)) )\
                                                                    {   (void)swprintf_s( &sDestString[__Sec_Strnlen(sDestString, nMaxLength)], __Sec_Remaining_strlen(sDestString, nMaxLength), sFormat, __VA_ARGS__ );    }\
                                                                    __Sec_CloseN( sDestString, nMaxLength )

// Add static string with defined length
#define __Sec_strncpy( sDestString, sSrcString, nMaxLength )        (void)wcsncpy_s( sDestString, sSrcString , __Sec_sizeofN(sDestString, nMaxLength) );\
                                                                    __Sec_CloseN( sDestString, nMaxLength )

// Add static string (until maximum allowed length)
#define __Sec_strcpy( sDestString, sSrcString )                     __Sec_strncpy( sDestString, sSrcString, __Sec_sizeof(sDestString) )

// Add static ASC string with defined length
#define __Sec_strncpy_asc( sDestString, sSrcString, nMaxLength )    strncpy_s( sDestString, sSrcString , __Sec_sizeofN(sDestString, nMaxLength) );\
                                                                    __Sec_CloseN( sDestString, nMaxLength )

// Add static ASC string (until maximum allowed length)
#define __Sec_strcpy_asc( sDestString, sSrcString )                 __Sec_strncpy_asc( sDestString, sSrcString, __Sec_sizeof(sDestString) )

// Macros and function for secure add text to a buffer
// ------------------------------------------------------------------------------------------------------


// ------------------------------------------------------------------------------------------------------
// Combine text generating and adding to MultiAnalyser
// Internal text buffer for generating text
wchar_t g_strTextBuffer[4096] = { 0 };

// Pointer for message data (use #undef this and renew #define for changeing)
#define __MSGDATA_POINTER                               pstMessageData

// Combine text generating and adding to MultiAnalyser
#define __Mas_PduOpen_sprintf( sLayer, sFormat, ... )   if ( (__MSGDATA_POINTER) && (__MSGDATA_POINTER)->callback_PduOpen )\
                                                        {   __Sec_Empty( g_strTextBuffer );\
                                                            __Sec_Add_sprintf( g_strTextBuffer, __Sec_sizeof(g_strTextBuffer), sFormat, __VA_ARGS__ );\
                                                            (__MSGDATA_POINTER)->callback_PduOpen( g_strTextBuffer, sLayer, (__MSGDATA_POINTER)->pInternalParameter );\
                                                        } else assert( !"__Mas_PduOpen_sprintf: No callback!" )

#define __Mas_PduText_sprintf( sFormat, ... )           if ( (__MSGDATA_POINTER) && (__MSGDATA_POINTER)->callback_PduAddText )\
                                                        {   __Sec_Empty( g_strTextBuffer );\
                                                            __Sec_Add_sprintf( g_strTextBuffer, __Sec_sizeof(g_strTextBuffer), sFormat, __VA_ARGS__ );\
                                                            (__MSGDATA_POINTER)->callback_PduAddText( g_strTextBuffer, (__MSGDATA_POINTER)->pInternalParameter );\
                                                        } else assert( !"__Mas_PduText_sprintf: No callback!" )

#define __Mas_PduText_strcpy( sString )                 if ( (__MSGDATA_POINTER) && (__MSGDATA_POINTER)->callback_PduAddText )\
                                                        {   (__MSGDATA_POINTER)->callback_PduAddText( sString, (__MSGDATA_POINTER)->pInternalParameter );\
                                                        } else assert( !"__Mas_PduText_strcpy: No callback!" )

#define __Mas_PduText_RawLine()                         __Mas_PduText_RawLine_Function( (__MSGDATA_POINTER)->ucData, (__MSGDATA_POINTER)->ulDataLength, (__MSGDATA_POINTER) );

#define __Mas_PduText_HexBinAsc( pData, nBytes )        __Mas_PduText_HexBinAsc_Function( (const unsigned char * const)pData, (const unsigned int)nBytes, __MSGDATA_POINTER )

#define __Mas_PduCose()                                 if ( (__MSGDATA_POINTER) && (__MSGDATA_POINTER)->callback_PduClose )\
                                                        {    (__MSGDATA_POINTER)->callback_PduClose( (__MSGDATA_POINTER)->pInternalParameter ); }\
                                                        else assert( !"__Mas_PduCose: No callback!" )

// Macros and function for secure add text to a buffer
#ifndef min
#define __Sec_Min( a, b )                                       (((a) < (b)) ? (a) : (b))
#else  // min
#define __Sec_Min                                               min
#endif // min
#ifndef max
#define __Sec_Max( a, b )                                       (((a) > (b)) ? (a) : (b))
#else  // min
#define __Sec_Max                                               max
#endif // min

#define __Sec_strrchr( sSourceA, nChar )                            ((intptr_t)((wcsrchr( sSourceA, nChar ))-(const wchar_t*)sSourceA))
#define __Sec_strncpy2( sDestString, sSrcString, nMaxLength )       (void)wcsncpy_s( sDestString, nMaxLength, sSrcString, _TRUNCATE );\
                                                                    (sDestString)[__Sec_Max(1,nMaxLength)-1] = 0

// Print hexadecimal data dump
static const wchar_t g_cAscDump[256] =
{ // Print char or replace by dummy
    L'.',L'.',L'.', L'.',L'.',L'.',L'.',L'.', L'.',L'.',L'.',L'.',L'.', L'.',L'.',L'.',
    L'.',L'.',L'.', L'.',L'.',L'.',L'.',L'.', L'.',L'.',L'.',L'.',L'.', L'.',L'.',L'.',
    L' ',L'!',L'\"',L'#',L'$',L'%',L'&',L'\'',L'(',L')',L'*',L'+',L',', L'-',L'.',L'/',
    L'0',L'1',L'2', L'3',L'4',L'5',L'6',L'7', L'8',L'9',L':',L';',L'<', L'=',L'>',L'?',
    L'@',L'A',L'B', L'C',L'D',L'E',L'F',L'G', L'H',L'I',L'J',L'K',L'L', L'M',L'N',L'O',
    L'P',L'Q',L'R', L'S',L'T',L'U',L'V',L'W', L'X',L'Y',L'Z',L'[',L'\\',L']',L'^',L'_',
    L'`',L'a',L'b', L'c',L'd',L'e',L'f',L'g', L'h',L'i',L'j',L'k',L'l', L'm',L'n',L'o',
    L'p',L'q',L'r', L's',L't',L'u',L'v',L'w', L'x',L'y',L'z',L'{',L'|', L'}',L'~',L'.',
    L'.',L'.',L'.', L'.',L'.',L'.',L'.',L'.', L'.',L'.',L'.',L'.',L'.', L'.',L'.',L'.',
    L'.',L'.',L'.', L'.',L'.',L'.',L'.',L'.', L'.',L'.',L'.',L'.',L'.', L'.',L'.',L'.',
    L'',L'',L'', L'',L'',L'',L'',L'', L'',L'',L'',L'',L'', L'.',L'',L'',
    L'',L'',L'', L'',L'',L'',L'',L'', L'',L'',L'',L'',L'', L'',L'',L'',
    L'',L'',L'', L'',L'',L'',L'',L'', L'',L'',L'',L'',L'', L'',L'',L'',
    L'',L'',L'', L'',L'',L'',L'',L'', L'',L'',L'',L'',L'', L'',L'',L'',
    L'',L'',L'', L'',L'',L'',L'',L'', L'',L'',L'',L'',L'', L'',L'',L'',
    L'',L'',L'', L'',L'',L'',L'',L'', L'',L'',L'',L'',L'', L'',L'',L''
};
void __Mas_PduText_HexBinAsc_Function( const unsigned char * const puData, const unsigned int ulBytes, const StructThirdPartyComponent5V_Data * const __MSGDATA_POINTER)
{
    unsigned char   ucByteCount = 0;
    unsigned int    ulIndex     = 0;
    //wchar_t   strLineText[119]    = { 0 }; __Sec_Empty( strLineText );
    wchar_t strHexText[25]      = { 0 }; __Sec_Empty( strHexText  );
    wchar_t strAscText[17]      = { 0 }; __Sec_Empty( strAscText  );
    wchar_t strBinText[73]      = { 0 }; __Sec_Empty( strBinText  ); wchar_t strBinShort[9] = { 0,0,0,0,0,0,0,0, 0 };

    // Add all bytes
    for ( unsigned int ulIndex = 0; (ulBytes) > ulIndex; ulIndex++ )
    { // Make diffrent data types, BIN, HEX and ASC for one BYTE
        for ( unsigned char ucBit = 7; 7 >= ucBit; ucBit-- )
        {   strBinShort[7-ucBit] = (0x01 & (((puData)[ulIndex]) >> ucBit)) ? (L'1') : (L'0');   }
        __Sec_Add_sprintf( strBinText, __Sec_sizeof(strBinText), L"%s "   , strBinShort );
        __Sec_Add_sprintf( strHexText, __Sec_sizeof(strHexText), L"%2.2X ", (puData)[ulIndex] );
        __Sec_Add_sprintf( strAscText, __Sec_sizeof(strAscText), L"%c "   , g_cAscDump[ (puData)[ulIndex] ] );
        ucByteCount++;
        // Need to print line?
        if ( 8 <= ucByteCount )
        { // Join line:
            __Mas_PduText_sprintf( L"%s| %s| %s|\n", strBinText, strHexText, strAscText );
            // Prepare for next line:
            __Sec_Empty( strHexText ); __Sec_Empty( strAscText ); __Sec_Empty( strBinText  ); ucByteCount = 0;
        }
    } // for ( unsigned int ulIndex = 0; (ulBytes) > ulIndex; ulIndex++ )
      // Have unfinisched line?
    if ( ucByteCount )
    { // Fill not used space with white space:
        for ( ; 8 > ucByteCount; ucByteCount++ )
        {   __Sec_Add_sprintf( strBinText, __Sec_sizeof(strBinText), L"         " ); __Sec_Add_sprintf( strHexText, __Sec_sizeof(strHexText), L"   " ); __Sec_Add_sprintf( strAscText, __Sec_sizeof(strAscText), L"  " );   }
        // Join line:
        __Mas_PduText_sprintf( L"%s| %s| %s|\n", strBinText, strHexText, strAscText );
    } // if ( ucByteCount )
}

// Print RAW-Line
static const wchar_t g_cHexNible[16] = { L'0',L'1',L'2',L'3',L'4',L'5',L'6',L'7',L'8',L'9',L'A',L'B',L'C',L'D',L'E',L'F' };
void __Mas_PduText_RawLine_Function( const unsigned char * const puData, const unsigned int ulBits, const StructThirdPartyComponent5V_Data * const __MSGDATA_POINTER)
{
    // Add all bytes
    __Sec_Empty( g_strTextBuffer );
    __Sec_Add_sprintf( g_strTextBuffer, __Sec_sizeof(g_strTextBuffer), L"RAW-Data (%u bits):", ulBits );

    unsigned int    ulWritePos = (unsigned int)__Sec_Strnlen(g_strTextBuffer, __Sec_sizeof(g_strTextBuffer));
    // How many bytes to write (not the last may needed to mask)
    const unsigned int ulBytes = (0 == ulBits) ? (0) : (__Sec_Min( ((ulBits+7) >> 3)-1, ((__Sec_sizeof(g_strTextBuffer)-(5+ulWritePos)) / 3)) );
    assert( ulBytes == ((0 == ulBits) ? (0) : (((ulBits+7) >> 3)-1)) ); // Detect low string place
    // Write nibble
    unsigned int ulIndex = 0;
    for ( ; ulBytes > ulIndex; ulIndex++ )
    {
        g_strTextBuffer[ulWritePos++] = L' ';
        g_strTextBuffer[ulWritePos++] = g_cHexNible[(0xF & (puData[ulIndex] >> 4))];
        g_strTextBuffer[ulWritePos++] = g_cHexNible[(0xF & (puData[ulIndex]     ))];
    }
    // Neet mask last byte (remove not present bits)
    const unsigned char ucLastByte = (puData[ulIndex]) & ((7 & ulBits) ? ( 0xFF << (8 - (7 & ulBits)) ) : (0xFF));
    g_strTextBuffer[ulWritePos++] = L' ';
    g_strTextBuffer[ulWritePos++] = g_cHexNible[(0xF & (ucLastByte >> 4))];
    g_strTextBuffer[ulWritePos++] = g_cHexNible[(0xF & (ucLastByte     ))];
    // Terminate
    g_strTextBuffer[ulWritePos++] = L'\n';
    g_strTextBuffer[ulWritePos  ] = 0;

    // Add line
    __Mas_PduText_strcpy( g_strTextBuffer );
}
// ------------------------------------------------------------------------------------------------------



// ------------------------------------------------------------------------------------------------------
// Value for message read position (use #undef this and renew #define for changeing)
#define __MAS_READ_INDEX            ulReadIndex
#define __MAS_READ_SUCCESS          bReadSuccess
#define __Mas_DEFINE_ulReadIndex    unsigned int __MAS_READ_INDEX = 0; bool __MAS_READ_SUCCESS = false
// Define this in example throw exception or jump with goto...
#define __MAS_READ_FAIL_ACTION      assert( !"Read more bits than present" );

// Read wanted amount of bits, call "__MAS_READ_FAIL_ACTION" if fail
#define __Mas_Read8(  nBits )       __Mas_Read_Bits8_Function(  nBits, (__MSGDATA_POINTER)->ucData, (__MSGDATA_POINTER)->ulDataLength, &(__MAS_READ_INDEX), &(__MAS_READ_SUCCESS) ); if ( !(__MAS_READ_SUCCESS) ) { __MAS_READ_FAIL_ACTION; }
#define __Mas_Read16( nBits )       __Mas_Read_Bits16_Function( nBits, (__MSGDATA_POINTER)->ucData, (__MSGDATA_POINTER)->ulDataLength, &(__MAS_READ_INDEX), &(__MAS_READ_SUCCESS) ); if ( !(__MAS_READ_SUCCESS) ) { __MAS_READ_FAIL_ACTION; }
#define __Mas_Read32( nBits )       __Mas_Read_Bits32_Function( nBits, (__MSGDATA_POINTER)->ucData, (__MSGDATA_POINTER)->ulDataLength, &(__MAS_READ_INDEX), &(__MAS_READ_SUCCESS) ); if ( !(__MAS_READ_SUCCESS) ) { __MAS_READ_FAIL_ACTION; }

// Read wanted amount of bits, user defined behaviour of "__MAS_READ_FAIL_ACTION"
#define __Mas_Watch8(  nBits )      __Mas_Read_Bits8_Function(  nBits, (__MSGDATA_POINTER)->ucData, (__MSGDATA_POINTER)->ulDataLength, &(__MAS_READ_INDEX), &(__MAS_READ_SUCCESS) )
#define __Mas_Watch16( nBits )      __Mas_Read_Bits16_Function( nBits, (__MSGDATA_POINTER)->ucData, (__MSGDATA_POINTER)->ulDataLength, &(__MAS_READ_INDEX), &(__MAS_READ_SUCCESS) )
#define __Mas_Watch32( nBits )      __Mas_Read_Bits32_Function( nBits, (__MSGDATA_POINTER)->ucData, (__MSGDATA_POINTER)->ulDataLength, &(__MAS_READ_INDEX), &(__MAS_READ_SUCCESS) )

// Read max 1 to 8 bits of data and return result, increase pulReadBitPos
unsigned char __Mas_Read_Bits8_Function( const unsigned char ucBits, const unsigned char *pucSource, const unsigned int ulMaxBitLength, unsigned int *pulReadBitPos, bool *pbSuccess )
{
    unsigned int        ulValue         = 0;
    unsigned int        ulBytePos       = ((*pulReadBitPos)  >> 3);
    const unsigned int  ulMaxBytePos    = (ulMaxBitLength+7) >> 3 ;
    unsigned char       ucDest          = 0;
    const unsigned char ucReverseBits   =  (8 - ucBits);
    const unsigned char ucBitPos        = ((8 - ((*pulReadBitPos) &  0x7)) + ucReverseBits);

    // Can read?
    if ( 8 < ucBits                                 )   {   assert( 8             >= ucBits                    );  if (pbSuccess) { (*pbSuccess) = false; } return 0;   }
    if ( ulMaxBitLength < ((*pulReadBitPos)+ucBits) )   {   assert( ulMaxBitLength >= ((*pulReadBitPos)+ucBits) ); if (pbSuccess) { (*pbSuccess) = false; } return 0;   }

    // Read data (16 bits, so every possible parts of 8 bits are included for 8 bit value)
                                        ulValue |=  ((unsigned int)pucSource[ulBytePos++] << 8);
    if ( ulMaxBytePos >= ulBytePos )    ulValue |=  ((unsigned int)pucSource[ulBytePos++]     );

    // Shift bits to LSB
    ucDest       = (unsigned char)(ulValue >> ucBitPos);
    // Mask only wanted bits
    ucDest      &= (0xFF >> ucReverseBits);

    // Increase bits
    (*pulReadBitPos)    += ucBits;

    if (pbSuccess) { (*pbSuccess) = true; }
    return ucDest;
}

// Read max 1 to 16 bits of data and return result, increase pulReadBitPos
unsigned short __Mas_Read_Bits16_Function( const unsigned char ucBits, const unsigned char *pucSource, const unsigned int ulMaxBitLength, unsigned int *pulReadBitPos, bool *pbSuccess)
{
    unsigned int        ulValue         = 0;
    unsigned int        ulBytePos       = ((*pulReadBitPos)  >> 3);
    const unsigned int  ulMaxBytePos    = (ulMaxBitLength+7) >> 3 ;
    unsigned short      usDest          = 0;
    const unsigned char ucReverseBits   = (16 - ucBits);
    const unsigned char ucBitPos        = ((8 - ((*pulReadBitPos) &  0x7)) + ucReverseBits);

    // Can read?
    if ( 16 < ucBits                                )   {   assert( 16             >= ucBits                    ); if (pbSuccess) { (*pbSuccess) = false; } return 0;   }
    if ( ulMaxBitLength < ((*pulReadBitPos)+ucBits) )   {   assert( ulMaxBitLength >= ((*pulReadBitPos)+ucBits) ); if (pbSuccess) { (*pbSuccess) = false; } return 0;   }

    // Read data (24 bits, so every possible parts of 8 bits are included for 16 bit value)
                                        ulValue |=  ((unsigned int)pucSource[ulBytePos++] << 16);
    if ( ulMaxBytePos >= ulBytePos )    ulValue |=  ((unsigned int)pucSource[ulBytePos++] <<  8);
    if ( ulMaxBytePos >= ulBytePos )    ulValue |=  ((unsigned int)pucSource[ulBytePos++]      );

    // Shift bits to LSB
    usDest       = (unsigned short)(ulValue >> ucBitPos);
    // Mask only wanted bits
    usDest      &= (0xFFFF >> ucReverseBits);

    // Increase bits
    (*pulReadBitPos)    += ucBits;

    if (pbSuccess) { (*pbSuccess) = true; }
    return usDest;
}

// Read max 1 to 32 bits of data and return result, increase pulReadBitPos
unsigned int __Mas_Read_Bits32_Function( const unsigned char ucBits, const unsigned char *pucSource, const unsigned int ulMaxBitLength, unsigned int *pulReadBitPos, bool *pbSuccess)
{
    unsigned long long  ullValue        = 0;
    unsigned int        ulDest          = 0;
    unsigned int        ulBytePos       = ((*pulReadBitPos)  >> 3);
    const unsigned int  ulMaxBytePos    = (ulMaxBitLength+7) >> 3 ;
    const unsigned char ucReverseBits   = (32 - ucBits);
    const unsigned char ucBitPos        = ((8 - ((*pulReadBitPos) &  0x7)) + ucReverseBits);

    // Can read?
    if ( 32 < ucBits                                )   {   assert( 32             >= ucBits                    ); if (pbSuccess) { (*pbSuccess) = false; } return 0;   }
    if ( ulMaxBitLength < ((*pulReadBitPos)+ucBits) )   {   assert( ulMaxBitLength >= ((*pulReadBitPos)+ucBits) ); if (pbSuccess) { (*pbSuccess) = false; } return 0;   }

    // Read data (40 bits, so every possible parts of 8 bits are included for 32 bit value)
                                        ullValue    |=  ((unsigned long long)pucSource[ulBytePos++] << 32);
    if ( ulMaxBytePos >= ulBytePos )    ullValue    |=  ((unsigned long long)pucSource[ulBytePos++] << 24);
    if ( ulMaxBytePos >= ulBytePos )    ullValue    |=  ((unsigned long long)pucSource[ulBytePos++] << 16);
    if ( ulMaxBytePos >= ulBytePos )    ullValue    |=  ((unsigned long long)pucSource[ulBytePos++] <<  8);
    if ( ulMaxBytePos >= ulBytePos )    ullValue    |=  ((unsigned long long)pucSource[ulBytePos++]      );

    // Shift bits to LSB
    ulDest       = (unsigned int)(ullValue >> ucBitPos);
    // Mask only wanted bits
    ulDest      &= (0xFFFFFFFF >> ucReverseBits);

    // Increase bits
    (*pulReadBitPos)    += ucBits;

    if (pbSuccess) { (*pbSuccess) = true; }
    return ulDest;
}
// ------------------------------------------------------------------------------------------------------


// ------------------------------------------------------------------------------------------------------
typedef enum EnumResultFilterSettings : unsigned long long
{ // Values for "StructThirdPartyComponent5V_Result::ullFilterMask"
    Filter_ViewAll                                  = 0,
    // ------------------------------------------------------------------------------------------------------------------------------------------------
    TetraTmo_Supress_Slot_1                         = 0x0000000000000001,   // Supress timeslot
    TetraTmo_Supress_Slot_2                         = 0x0000000000000002,   // Supress timeslot
    TetraTmo_Supress_Slot_3                         = 0x0000000000000004,   // Supress timeslot
    TetraTmo_Supress_Slot_4                         = 0x0000000000000008,   // Supress timeslot

    TetraTmo_Supress_DL_NullPdu                     = 0x0000000000000010,   // Suppress NULL-PDU as end of data in PDU
    TetraTmo_Supress_DL_EqualBroadCast              = 0x0000000000000020,   // SYNC, SYSINFO(-Q/DA), ACCESS-DEFINE
    TetraTmo_Supress_DL_EqualDnwrk                  = 0x0000000000000040,   // D-NWRK(-DA), D-NWRK Extension
    TetraTmo_Supress_DL_NotDecoded                  = 0x0000000000000080,   // Supress physical layer information
    TetraTmo_Supress_DL_USignal                     = 0x0000000000000100,   // Supress MAC-U-SIGNAL on FACCH
    TetraTmo_Supress_DL_ErrorInPdu                  = 0x0000000000001000,   // Supress PDUs if an error occured

    TetraTmo_Supress_UL_NullPdu                     = 0x0000000000000200,   // Suppress NULL-PDU as end of data in PDU (not stand alone NULL-PDU!)
    TetraTmo_Supress_UL_NotDecodedBelowThreshhold   = 0x0000000000000400,   // Supress physical layer information, if they are below RSSI treshhold
    TetraTmo_Supress_UL_USignal                     = 0x0000000000000800,   // Supress MAC-U-SIGNAL on FACCH
    TetraTmo_Supress_UL_ErrorInPdu                  = 0x0000000000002000,   // Supress PDUs if an error occured

    TetraTmo_Supress_UserDefined_1                  = 0x0000000000010000,   // Usage from external DLL analysis
    TetraTmo_Supress_UserDefined_2                  = 0x0000000000020000,   // Usage from external DLL analysis
    // ------------------------------------------------------------------------------------------------------------------------------------------------
    TetraDmo_Supress_Slot_1                         = 0x0000000000000001,   // Supress timeslot
    TetraDmo_Supress_Slot_2                         = 0x0000000000000002,   // Supress timeslot
    TetraDmo_Supress_Slot_3                         = 0x0000000000000004,   // Supress timeslot
    TetraDmo_Supress_Slot_4                         = 0x0000000000000008,   // Supress timeslot

    TetraDmo_Supress_NoCall_Slots                   = 0x0000000000000010,   // Supress timeslot
    TetraDmo_Supress_Master_Slots                   = 0x0000000000000020,   // Supress timeslot
    TetraDmo_Supress_Slave_Slots                    = 0x0000000000000040,   // Supress timeslot

    TetraDmo_Supress_NotDecodedBelowThreshhold      = 0x0000000000001000,   // Supress physical layer information
    TetraDmo_Supress_Repeated                       = 0x0000000000002000,   // Supress repeated messages(frame count down)
    TetraDmo_Supress_LateEntry                      = 0x0000000000004000,   // Supress DM-OCCPIED and DM-RESERVERD
    TetraDmo_Supress_USignal                        = 0x0000000000008000,   // Supress MAC-U-SIGNAL
    TetraDmo_Supress_Idle_DPresSync                 = 0x0000000000010000,   // Supress DPres sync with IDLE channel state
    TetraDmo_Supress_ErrorInPdu                     = 0x0000000000020000,   // Supress PDUs if an error occured
    // ------------------------------------------------------------------------------------------------------------------------------------------------
    Dmr_Supress_Slot_1                              = 0x0000000000000001,   // Supress timeslot
    Dmr_Supress_Slot_2                              = 0x0000000000000002,   // Supress timeslot

    Dmr_Supress_DL_IDLE_Pdu                         = 0x0000000000000010,   // Suppress IDLE-PDU
    Dmr_Supress_DL_EqualBroadCast                   = 0x0000000000000020,   // Suppress redunant broadcast
    Dmr_Supress_DL_NotDecoded                       = 0x0000000000000040,   // Supress physical layer information
    Dmr_Supress_DL_L2_CH_INFO                       = 0x0000000000000080,   // Supress SlotType/EMB/TACT if no higher layer present
    Dmr_Supress_DL_LateEntry                        = 0x0000000000000100,   // Supress late entry information in embedded channel
    Dmr_Supress_DL_ErrorInPdu                       = 0x0000000000100000,   // Supress PDUs if an error occured

    Dmr_Supress_UL_IDLE_Pdu                         = 0x0000000000001000,   // Suppress IDLE-PDU
    Dmr_Supress_UL_EqualBroadCast                   = 0x0000000000002000,   // Suppress redunant broadcast
    Dmr_Supress_UL_NotDecoded                       = 0x0000000000004000,   // Supress physical layer information, if they are below RSSI treshhold
    Dmr_Supress_UL_L2_CH_INFO                       = 0x0000000000008000,   // Supress SlotType/EMB/TACT if no higher layer present
    Dmr_Supress_UL_LateEntry                        = 0x0000000000010000,   // Supress late entry information in embedded channel
    Dmr_Supress_UL_ErrorInPdu                       = 0x0000000000200000    // Supress PDUs if an error occured
    // ------------------------------------------------------------------------------------------------------------------------------------------------
} EnumResultFilterSettings;
// ------------------------------------------------------------------------------------------------------


// ------------------------------------------------------------------------------------------------------
// DLL instances
#define DLL_INSTANCE_MAGIC      (0x3746723823734826)


struct StructSsiList
{
    bool            bWhiteList;
    unsigned int    ulSsiCount;
    unsigned int    ulSsi[100];
};
struct StructSsiCfg
{
    struct StructSsiList    stSsiList[2];
};

// Instance data
typedef struct StructDllInstance
{
    StructDllInstance()
    {
        ullMagic                = DLL_INSTANCE_MAGIC;
        pstNext                 = nullptr;
        memset( &strMafFileName , 0, sizeof(strMafFileName) );
        ucOpenCfg               = 0;
        memset( &stCfg          , 0, sizeof(stCfg) );
    }
    ~StructDllInstance()
    {
        ullMagic                = 0;
        pstNext                 = nullptr;
        memset( &strMafFileName , 0, sizeof(strMafFileName) );
        ucOpenCfg               = 0;
        memset( &stCfg          , 0, sizeof(stCfg) );
    }
    // Pointer error detction
    unsigned long long   ullMagic;  // DLL_INSTANCE_MAGIC

    // Multi Instance support by chain list
    StructDllInstance   *pstNext;

    wchar_t              strMafFileName[_MAX_PATH];

    unsigned char        ucOpenCfg = 0; // 0=None, 1=Have CFG, 2=Failed
    StructSsiCfg         stCfg;
} StructDllInstance;
StructDllInstance   *g_pstDllInstance = NULL;

// Clean up DLL instance
static int Raw_Delete_DllInstance( StructDllInstance *pstDll )
{
    StructDllInstance   *pstDllTemp = g_pstDllInstance, *pstLast = NULL;

    // Find DLL instance
    if ( NULL == pstDll || (DLL_INSTANCE_MAGIC != pstDll->ullMagic) )
    {   return -1;  }

    // Find instance in chain list
    while ( pstDllTemp )
    {
        assert( DLL_INSTANCE_MAGIC == pstDllTemp->ullMagic );
        // Found?
        if ( pstDllTemp == pstDll )
        { // Remove from chain list
            if ( pstLast )  {   pstLast->pstNext = pstDll->pstNext; }
            else            {   g_pstDllInstance = pstDll->pstNext; }
            break;
        }
        // Check next
        pstLast     = pstDllTemp;
        pstDllTemp  = pstDllTemp->pstNext;
    } // while ( pstDllTemp )

    delete pstDll;

    return 0;
}

// Delete all instances
int Raw_Delete_All_DllInstances()
{
    // Find instance in chain list
    while ( g_pstDllInstance )
    {
        if ( Raw_Delete_DllInstance(g_pstDllInstance) )
        {   return -1;  }
    } // while ( g_pstDllInstance )

    return 0;
}
// ------------------------------------------------------------------------------------------------------

// ------------------------------------------------------------------------------------------------------
const wchar_t *g_pstrOpen_Ansi      = L"rtS, ccs=UTF-8"; // _T("rtS") (UTF8 seem to be not to make problems, and will support UTF8 without BOM)
const wchar_t *g_pstrOpen_UTF8_BOM  = L"rtS, ccs=UTF-8";
const wchar_t *g_pstrOpen_UTF16_BOM = L"rtS, ccs=UTF-16LE";

const struct StructCfgKey
{
    const wchar_t   *pstrKey;
    const size_t     tKeySize;
} g_stKeys[] =
{
    {   L"SSI="      , wcsnlen(L"SSI="      ,80) },
    {   L"SECTION="  , wcsnlen(L"SECTION="  ,80) },
    {   L"LIST_TYPE=", wcsnlen(L"LIST_TYPE=",80) }
};
const bool Raw_ReadCfgFile( const wchar_t *pstrFileName, int *plErrno, StructSsiCfg *pstCfg )
{
    wchar_t          strLine[80]; memset(strLine, 0, sizeof(strLine));
    unsigned char    ucBom[4]   = { 0, 0, 0, 0 };
    const wchar_t   *pstrOpen   = g_pstrOpen_Ansi;
    FILE            *pFile      = nullptr;
    StructSsiList   *pstSsiList = nullptr;
    errno_t          tError     = 0;
    bool             bRc        = false;

    // Clean all
    if ( nullptr == pstCfg )
    {   goto Raw_GetReadFlag_End;   }
    memset(pstCfg, 0, sizeof(pstCfg[0]));

    // ---------------------------------------------------------------------------------------------------------------------------------
    // Open file for reading BOM
    tError = _wfopen_s( &pFile, pstrFileName, L"rb" );
    if ( NULL == pFile )
    {   if ( plErrno ) { (*plErrno) = tError; } goto Raw_GetReadFlag_End;   }
    // Read BOOM
    if ( sizeof(ucBom) != fread(ucBom, 1, sizeof(ucBom), pFile) )
    {   if ( plErrno ) { (*plErrno) = STRUNCATE; } goto Raw_GetReadFlag_End;    }
    // Detect type
    if ( ((0xEF == ucBom[0]) && (0xBB == ucBom[1]) && (0xBF == ucBom[2])                                )   )   {   pstrOpen = g_pstrOpen_UTF8_BOM; }   else    // UTF-8
    if ( ((0xFF == ucBom[0]) && (0xFE == ucBom[1])                                                      )   )   {   pstrOpen = g_pstrOpen_UTF16_BOM; }  else    // UTF-16 (LE)
    if ( ((0xFE == ucBom[0]) && (0xFF == ucBom[1])                                                      ) ||                                                    // UTF-16 (BE)
         ((0x00 == ucBom[0]) && (0x00 == ucBom[1]) && (0xFF == ucBom[2]) && (0xFF == ucBom[3])          ) ||                                                    // UTF-32 (BE)
         ((0xFF == ucBom[0]) && (0xFF == ucBom[1]) && (0x00 == ucBom[2]) && (0x00 == ucBom[3])          ) ||                                                    // UTF-32 (LE)
         ((0xF7 == ucBom[0]) && (0x64 == ucBom[1]) && (0x4C == ucBom[2])                                ) ||                                                    // UTF-1
         ((0x2B == ucBom[0]) && (0x2F == ucBom[1]) && (0x76 == ucBom[2]) && (0x38 == (0xFC & ucBom[3])) ) ||                                                    // UTF-7
         ((0xDD == ucBom[0]) && (0x73 == ucBom[1]) && (0x66 == ucBom[2]) && (0x73 == ucBom[3])          ) ||                                                    // UTF-EBCDIC
         ((0x0E == ucBom[0]) && (0xFE == ucBom[1]) && (0xFF == ucBom[2])                                ) ||                                                    // SCSU
         ((0xFB == ucBom[0]) && (0xEE == ucBom[1]) && (0x28 == ucBom[2])                                ) ||                                                    // BOCU-1
         ((0x84 == ucBom[0]) && (0x31 == ucBom[1]) && (0x95 == ucBom[2]) && (0x33 == ucBom[3])          )                                                       // GB 18030
       )
    {   if ( plErrno ) { (*plErrno) = ENOSYS; } goto Raw_GetReadFlag_End;   }
    fclose(pFile); pFile= NULL;
    // ---------------------------------------------------------------------------------------------------------------------------------
    // Open now file in correct text-mode...
    tError = _wfopen_s( &pFile, pstrFileName, pstrOpen );
    if ( NULL == pFile )
    {   if ( plErrno ) { (*plErrno) = tError; }    goto Raw_GetReadFlag_End;    }
    // ---------------------------------------------------------------------------------------------------------------------------------
    // Read configurion
    while( 1 == fwscanf_s( pFile, L"%79s\n", strLine, (unsigned int)_countof(strLine) ) )
    {
        strLine[ARRAYSIZE(strLine)-1] = 0;
        // Check for keywords
        for ( unsigned int ulKeyWord = 0; ARRAYSIZE(g_stKeys) > ulKeyWord; ulKeyWord++ )
        {
            if ( 0 != _wcsnicmp(strLine, g_stKeys[ulKeyWord].pstrKey, g_stKeys[ulKeyWord].tKeySize) )
            {   continue;   }

            switch ( ulKeyWord )
            {
                case 0: // "SSI="
                {   if ( !pstSsiList )
                    {   break;  }
                    // Get SSI and check valid range
                    const int lSsi = _wtoi( &strLine[g_stKeys[ulKeyWord].tKeySize] );
                    if ( (1 > lSsi) || (0x00FFFFFF < lSsi) )
                    {   break;  }
                    // Can add?
                    if ( ARRAYSIZE(pstSsiList->ulSsi) > pstSsiList->ulSsiCount )
                    {   pstSsiList->ulSsi[ pstSsiList->ulSsiCount ] = (unsigned int)lSsi; pstSsiList->ulSsiCount++; }
                }
                break;
                case 1: // "SECTION="
                {
                    const int lSection = _wtoi( &strLine[g_stKeys[ulKeyWord].tKeySize] );
                    if ( 1 > lSection || ARRAYSIZE(pstCfg->stSsiList) < lSection )
                    {   pstSsiList  = nullptr;  }
                    else
                    {   pstSsiList  = &pstCfg->stSsiList[lSection-1];   }
                }
                break;
                case 2: // "LIST_TYPE="
                {   if ( !pstSsiList )
                    {   break;  }
                    // White or black list
                    pstSsiList->bWhiteList = (0 < _wtoi(&strLine[g_stKeys[ulKeyWord].tKeySize]));
                }
                break;
            } // switch ( ulKeyWord )

            break;
        } // for ( unsigned int ulKeyWord = 0; ARRAYSIZE(stKeys) > ulKeyWord; ulKeyWord++ )
    } // while( 1 == fwscanf_s( pFile, L"%79s\n", strLine, (unsigned int)_countof(strLine) ) )
    // ---------------------------------------------------------------------------------------------------------------------------------
    // Have SSIs in list?
    for ( unsigned int ulList = 0; ARRAYSIZE(pstCfg->stSsiList) > ulList; ulList++ )
    {   if ( pstCfg->stSsiList[ulList].ulSsiCount ) {   bRc= true; break;   }   }
    // ---------------------------------------------------------------------------------------------------------------------------------

Raw_GetReadFlag_End:
    if ( NULL != pFile )
    {   fclose(pFile); pFile= NULL; }
    return bRc;
}

// Parse path+file+suffix into subparts: Path
bool Raw_GetPathName( const wchar_t *pstrPathFile, wchar_t *pstrPath, const unsigned int ulMaxPathLength )
{
    // Find last "/" or "\"
    const intptr_t lPosition    = max( __Sec_strrchr(pstrPathFile, '/'), __Sec_strrchr(pstrPathFile, '\\') );

    // Clean string
    __Sec_Empty( pstrPath );

    // Have path?
    if ( (0 > lPosition) )
    {   return  false;  }

    // Copy path
    __Sec_strncpy2( pstrPath, pstrPathFile, min((size_t)(lPosition+2),(size_t)ulMaxPathLength) );

    return (0 != pstrPath[0]);
}
// Read configuration for export
bool Raw_Open_Configuration( StructDllInstance *pstDll )
{
    wchar_t strFileName[_MAX_PATH] = { 0 };
    FILE *pFileOpen = NULL;

    // Open configuration file in the path
    if ( Raw_GetPathName(pstDll->strMafFileName, strFileName, _MAX_PATH) && strFileName[0] )
    { // Add filename and open
        __Sec_Add_sprintf( strFileName, _MAX_PATH, L"\\MultiAnalyzerSsiFilter.txt" );
        if ( true == Raw_ReadCfgFile(strFileName, nullptr, &pstDll->stCfg) )
        {   pstDll->ucOpenCfg = 1; return true; }
    } // if ( Raw_GetPathName(pstDll->strMafFileName, strFileName, _MAX_PATH) && strFileName[0] )
    // Not open try on desktop
    if ( NULL == pFileOpen )
    { // Get Desktop
        if ( SHGetSpecialFolderPath(HWND_DESKTOP, strFileName, CSIDL_DESKTOP, FALSE) && strFileName[0] )
        { // Add filename and open
            __Sec_Add_sprintf( strFileName, _MAX_PATH, L"\\MultiAnalyzerSsiFilter.txt" );
            if ( true == Raw_ReadCfgFile(strFileName, nullptr, &pstDll->stCfg) )
            {   pstDll->ucOpenCfg = 1; return true; }
        }
    } // if ( NULL == pFileOpen )
      // Not open try some other pathes
    if ( NULL == pFileOpen )
    {
        const wchar_t *pstrCfgDirs[] =
        {
            L"USERPROFILE",
            L"APPDATA",
            L"LOCALAPPDATA",
            L"ProgramFiles" // Special one (install drawer)
        };
        // Try all
        for ( unsigned int ulKeyWord = 0; ARRAYSIZE(pstrCfgDirs) > ulKeyWord; ulKeyWord++ )
        { // Get more possiblities
            strFileName[0] = 0; wchar_t *pstrPath = strFileName; size_t tBufferLength = _MAX_PATH;
            if ( _wdupenv_s( &pstrPath, &tBufferLength, pstrCfgDirs[ulKeyWord] ) || (nullptr == pstrPath) || (0 == pstrPath[0]) )
            {   continue;   }
            // Add filename and open
            if ( (ARRAYSIZE(pstrCfgDirs)-1) == ulKeyWord )
            {   __Sec_Add_sprintf( strFileName, _MAX_PATH, L"%s\\MultiAnalyzer\\MultiAnalyzerSsiFilter.txt", pstrPath ); }
            else
            {   __Sec_Add_sprintf( strFileName, _MAX_PATH, L"%s\\MultiAnalyzerSsiFilter.txt", pstrPath );    }
            if ( true == Raw_ReadCfgFile(strFileName, nullptr, &pstDll->stCfg) )
            {   pstDll->ucOpenCfg = 1; return true; }
        } // for ( unsigned int ulKeyWord = 0; ARRAYSIZE(pstrCfgDirs) > ulKeyWord; ulKeyWord++ )
    } // if ( NULL == pFileOpen )

    // Failure
    pstDll->ucOpenCfg = 2;
    return false;
}

// ------------------------------------------------------------------------------------------------------

// DLL provides wanted types of data for it own analysing...
// Returns 0: If successfull, all other values marking an error
THIRD_PARTY_COMPONENT_5V int ThirdPartyComponent5V_Init( StructThirdPartyComponent5V_Init *pstRegisterMessages )
{
    // Have result struct?
    if ( NULL == pstRegisterMessages )
    {   return -1;  }

    // Valid result struct?
    switch ( pstRegisterMessages->ulVersion )
    {
        case 1: // First version of struct
        case 2: // Extended version of version 1 struct
        break;

        default:
        return -2;
    } //  switch ( pstRegisterMessages->ulVersion )

    // Internal init
    __Sec_Empty( g_strTextBuffer );

    // Version/Company/Copyright string
    __Sec_strcpy_asc( pstRegisterMessages->strVersion   , "ThirdPartyComponent_SsiFilter5V, Version 1.0.0.0" );

    // Supported Protokoll, which type of protokoll data is supported
    pstRegisterMessages->ulProtokollSupport     = ThirdPartyComponent5V_TetraTmo;
    pstRegisterMessages->pDllInstance           = NULL;

    // All registred data
    pstRegisterMessages->ulRegisterCount        = 0;
    memset( pstRegisterMessages->stRegisterData , 0 , sizeof(pstRegisterMessages->stRegisterData) );

    StructThirdPartyComponent5V_RegData *pstRegData = NULL;
    //// --------------------------------------------------------------------
    // Set data: Analyse registred PDU* NOTE1                   ulType: UMac-PDU Type* NOTE2        , ulExtendedType: -  or "MAC-FRAG, MAC-END, D/U-BLCK" (0xFFFFFFFF=ALL)
    pstRegData = &pstRegisterMessages->stRegisterData[ pstRegisterMessages->ulRegisterCount ];
    pstRegData->eRegisterType     = ThirdPartyComponent5V_UMac_Pdu;
    pstRegData->ulDirection       = 0x03;     // DL+UL
    pstRegData->ulType            = UINT_MAX; // ALL
    pstRegData->ulExtendedType[0] = UINT_MAX; // ALL
    pstRegData->ulExtendedType[1] = UINT_MAX; // ALL
    pstRegisterMessages->ulRegisterCount++;

    // --------------------
    // Make new instance
    StructDllInstance   *pstDll = new (std::nothrow)StructDllInstance();
    if ( NULL == pstDll )
    {   return -1;  }

    // Add into chain list
    pstDll->pstNext     = g_pstDllInstance;
    g_pstDllInstance    = pstDll;

    // Store instance
    pstRegisterMessages->pDllInstance   = pstDll;
    // --------------------

    return 0;
}

// Multianalyser provides data for analysing:
// Input : StructThirdPartyComponent5V_Data  : The data themself for external analysing
// Output: StructThirdPartyComponent5V_Result: What is todo for multianalyser.
// Returns 0: Result is stored in StructThirdPartyComponent5V_Result, all other values are errors
THIRD_PARTY_COMPONENT_5V int ThirdPartyComponent5V_Data( void *pDllInstance, const StructThirdPartyComponent5V_Data *pstMessageData, StructThirdPartyComponent5V_Result *pstAnalyseResult )
{
    StructDllInstance   *pstDll = (StructDllInstance*)pDllInstance;

    // Have result struct?
    if ( NULL == (__MSGDATA_POINTER))
    {   return -1;  }

    // Have result struct?
    if ( NULL == pstAnalyseResult)
    {   return -2;  }

    // Valid result struct?
    if ( STRUCT_THIRDPARTYCOMPONENT5V_DATA__VERSION != (__MSGDATA_POINTER)->ulVersion )
    {   return -3;  }

    // Valid data?
    if ( STRUCT_THIRDPARTYCOMPONENT5V_RESULT__VERSION != pstAnalyseResult->ulVersion )
    {   return -4;  }

    // DLL instance
    if ( nullptr == pstDll || (DLL_INSTANCE_MAGIC != pstDll->ullMagic) )
    {   return -5;  }

    //Config status
    switch ( pstDll->ucOpenCfg )
    {
        case 0: // No CFG
            // Valid result struct?
            switch ( (__MSGDATA_POINTER)->ulVersion )
            { // Known versions
                case STRUCT_THIRDPARTYCOMPONENT5V_DATA__VERSION_V1:
                break;
                case STRUCT_THIRDPARTYCOMPONENT5V_DATA__VERSION_V2:
                case STRUCT_THIRDPARTYCOMPONENT5V_DATA__VERSION_V3:
                    // Information about MAF file
                    if ( (((const StructThirdPartyComponent5V_DataV2*)(__MSGDATA_POINTER))->bNewMafFile) || (0 == pstDll->strMafFileName[0]) )
                    { // Store file name
                        __Sec_strcpy(  pstDll->strMafFileName, ((const StructThirdPartyComponent5V_DataV2*)(__MSGDATA_POINTER))->pstrCurrentMafFileName );
                    }
                break;
                default:
                return -3;
            } // switch (  (__MSGDATA_POINTER)->ulVersion )

            if ( false == Raw_Open_Configuration( pstDll ) )
            {   return -1;  }
        break;
        case 1: // Configuration present
        break;
        default: // Failed to load
        return -6;
    } // switch ( pstDll->ucOpenCfg )

    // Ensure nothing returned
    pstAnalyseResult->eResult           = ThirdPartyComponent5V_Result_Nothing;
    pstAnalyseResult->ullFilterMask     = EnumResultFilterSettings::Filter_ViewAll;
    pstAnalyseResult->ulAnalyseLength   = 0;

    __Mas_DEFINE_ulReadIndex;   // unsigned int ulReadIndex = 0

    // Depending from message type write text for MSC
    switch ((__MSGDATA_POINTER)->eRegisterType )
    {
        case ThirdPartyComponent5V_UMac_Pdu:            // Analyse registred PDU* NOTE1                 ulType: UMac-PDU Type* NOTE2        , ulExtendedType: -  or "MAC-FRAG, MAC-END, D/U-BLCK" (0xFFFFFFFF=ALL)
        {
            unsigned int ulAddress = UINT_MAX;
            if ( (1 == (__MSGDATA_POINTER)->ulDirection)  && ((__MSGDATA_POINTER)->eTerminatorAddressType) )
            {   ulAddress = (__MSGDATA_POINTER)->ulTerminatorAddress;   }
            else
            if ( (2 == (__MSGDATA_POINTER)->ulDirection)  && ((__MSGDATA_POINTER)->eOrginatorAddressType) )
            {   ulAddress = (__MSGDATA_POINTER)->ulOrginatorAddress;    }
            else
            {   break;  }

            // Test for SSI
            for ( unsigned ucCfg = 0; ARRAYSIZE(pstDll->stCfg.stSsiList) > ucCfg; ucCfg++ )
            {
                const unsigned long long ullFilter = ((0 == ucCfg) ? (EnumResultFilterSettings::TetraTmo_Supress_UserDefined_1) : (EnumResultFilterSettings::TetraTmo_Supress_UserDefined_2));
                const StructSsiList& stSsiList = pstDll->stCfg.stSsiList[ucCfg];
                bool  bFoundSssi = false;
                for ( unsigned int ulCount = 0; stSsiList.ulSsiCount > ulCount; ulCount++ )
                {
                    if ( stSsiList.ulSsi[ulCount] == ulAddress )
                    {   bFoundSssi = true; break;       }
                }

                if ( true == bFoundSssi )
                { // SSI found, so filter if in blacklist
                    if ( false == stSsiList.bWhiteList )
                    {   pstAnalyseResult->ullFilterMask |= ullFilter; pstAnalyseResult->eResult = ThirdPartyComponent5V_Result_NothingButFilter;    }
                }
                else
                { // SSI NOT found, so filter if not in white list
                    if ( true == stSsiList.bWhiteList )
                    {   pstAnalyseResult->ullFilterMask |= ullFilter; pstAnalyseResult->eResult = ThirdPartyComponent5V_Result_NothingButFilter;    }
                }
            } // for ( unsigned ucCfg = 0; ARRAYSIZE(pstDll->stCfg.stSsiList) > ucCfg; ucCfg++ )

            //if ( 3   != (__MSGDATA_POINTER)->ulOrginatorAddress )
            //{ pstAnalyseResult->ullFilterMask |= EnumResultFilterSettings::TetraTmo_Supress_UserDefined_2; pstAnalyseResult->eResult = ThirdPartyComponent5V_Result_NothingButFilter; }
            //__Mas_PduCose();

            }
        break;

        default:
        break;
    } // switch ( (__MSGDATA_POINTER)->eRegisterType )

    // Result in "pstAnalyseResult" stored
    return 0;
}

// Called if no further data will be analysed
THIRD_PARTY_COMPONENT_5V int ThirdPartyComponent5V_Close(void *pDllInstance)
{
    StructDllInstance   *pstDll = (StructDllInstance*)pDllInstance, *pstDllTemp = g_pstDllInstance;

    // Delete instance
    (int)Raw_Delete_DllInstance( (StructDllInstance*)pDllInstance );

    // Nothing todo yet
    return 0;
}
