#include "RFC1144.h"
#include <Winsock2.h>           // hton, ntohl
#pragma comment(lib, "Ws2_32.lib")

/*
 * Routines to compress and uncompess tcp packets (for transmission
 * over low speed serial lines.
 *
 * Copyright (c) 1989 Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms are permitted
 * provided that the above copyright notice and this paragraph are
 * duplicated in all such forms and that any documentation,
 * advertising materials, and other materials related to such
 * distribution and use acknowledge that the software was developed
 * by the University of California, Berkeley.  The name of the
 * University may not be used to endorse or promote products derived
 * from this software without specific prior written permission.
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
 * WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 *
 * Van Jacobson (van@helios.ee.lbl.gov), Dec 31, 1989:
 *   Initial distribution.
 *
 * Modified June 1993 by Paul Mackerras, paulus@cs.anu.edu.au,
 * so that the entire packet being decompressed doesn't have
 * to be in contiguous memory (just the compressed header).
 *
 * Modified March 1998 by Guy Lancaster, glanca@gesn.com,
 * for a 16 bit processor.
 *
 * Modified September 2019 by Gunter Hinrichsen
 * Some small fixes... (i.e. Buffer underflow)
 */

typedef unsigned char   u_char;
typedef unsigned short  u_short;
//typedef unsigned long u_int;


/*
 * Structure of an internet header, naked of options.
 *
 * We declare ip_len and ip_off to be short, rather than u_short
 * pragmatically since otherwise unsigned comparisons can result
 * against negative integers quite easily, and fail in subtle ways.
 */
__pragma( pack(push, 1) )

struct ip
{
//#if BYTE_ORDER == LITTLE_ENDIAN
    u_char  ip_hl:4,        /* header length */
        ip_v:4;             /* version */
//#endif
//#if BYTE_ORDER == BIG_ENDIAN
//  u_char  ip_v[4],        /* version */
//      ip_hl[4];           /* header length */
//#endif
    u_char  ip_tos;         /* type of service */
    short   ip_len;         /* total length */
    u_short ip_id;          /* identification */
    short   ip_off;         /* fragment offset field */
#define IP_DF 0x4000        /* dont fragment flag */
#define IP_MF 0x2000        /* more fragments flag */
    u_char  ip_ttl;         /* time to live */
    u_char  ip_p;           /* protocol */
    u_short ip_sum;         /* checksum */
    struct  in_addr ip_src,ip_dst;  /* source and dest address */
};
#define     IPPROTO_TCP   6


#define TH_FIN  0x01
#define TH_SYN  0x02
#define TH_RST  0x04
#define TH_PUSH 0x08
#define TH_ACK  0x10
#define TH_URG  0x20
#define TH_FLAGS (TH_FIN | TH_SYN | TH_RST | TH_ACK | TH_URG)

typedef struct tcphdr {
    u_short th_sport;
    u_short th_dport;
    u_int  th_seq;
    u_int  th_ack;
//#ifndef __BIG_ENDIAN__
    u_char  th_x2:4,
            th_off:4;
//#else /* #ifndef __BIG_ENDIAN__ */
//    u_char  th_off:4,
//            th_x2:4;
//#endif
    u_char  th_flags;
    u_short th_win;
    u_short th_sum;
    u_short th_urp;
} tcphdr;

__pragma( pack(pop) )

#define MAX_STATES 16   /* must be >2 and <255 */
#define MAX_HDR 128     /* max TCP+IP hdr length (by protocol def) */

/* packet types */
#define TYPE_IP 0x40
#define TYPE_UNCOMPRESSED_TCP 0x70
#define TYPE_COMPRESSED_TCP 0x80
#define TYPE_ERROR 0x00 /* this is not a type that ever appears on
                        * the wire.  The receive framer uses it to
                        * tell the decompressor there was a packet
                        * transmission error. */
/*
* Bits in first octet of compressed packet
*/

/* flag bits for what changed in a packet */

#define NEW_C  0x40
#define NEW_I  0x20
#define TCP_PUSH_BIT 0x10

#define NEW_S  0x08
#define NEW_A  0x04
#define NEW_W  0x02
#define NEW_U  0x01

/* reserved, special-case values of above */
#define SPECIAL_I (NEW_S|NEW_W|NEW_U)        /* echoed interactive traffic */
#define SPECIAL_D (NEW_S|NEW_A|NEW_W|NEW_U)  /* unidirectional data */
#define SPECIALS_MASK (NEW_S|NEW_A|NEW_W|NEW_U)

/*
* "state" data for each active tcp conversation on the wire.  This is
* basically a copy of the entire IP/TCP header from the last packet together
* with a small identifier the transmit & receive ends of the line use to
* locate saved header.
*/

__declspec(align(8)) struct cstate {
    struct cstate *cs_next;  /* next most recently used cstate (xmit only) */
    u_short cs_hlen;         /* size of hdr (receive only) */
    u_char cs_id;            /* connection # associated with this state */
    u_char cs_filler;
    union {
            char csu_hdr[MAX_HDR];
            struct ip csu_ip;   /* ip/tcp hdr from most recent packet */
    } slcs_u;
};

#define cs_ip slcs_u.csu_ip
#define cs_hdr slcs_u.csu_hdr

/*
* all the state data for one serial line (we need one of these per line).
*/
struct slcompress {
    struct cstate *last_cs;            /* most recently used tstate */
    u_char last_recv;                  /* last rcvd conn. id */
    u_char last_xmit;                  /* last sent conn. id */
    u_short flags;
    struct cstate tstate[MAX_STATES];  /* xmit connection states */
    struct cstate rstate[MAX_STATES];  /* receive connection states */
};

/* flag values */
#define SLF_TOSS 1       /* tossing rcvd frames because of input err */

/*
* The following macros are used to encode and decode numbers.  They all
* assume that `cp' points to a buffer where the next byte encoded (decoded)
* is to be stored (retrieved).  Since the decode routines do arithmetic,
* they have to convert from and to network byte order.
*/

/*
* ENCODE encodes a number that is known to be non-zero.  ENCODEZ checks for
* zero (zero has to be encoded in the long, 3 byte form).
*/
#define ENCODE(n) { \
    if ((u_short)(n) >= 256) { \
            *cp++ = 0; \
            cp[1] = (n); \
            cp[0] = (n) >> 8; \
            cp += 2; \
    } else { \
            *cp++ = (n); \
    } \
}
#define ENCODEZ(n) { \
    if ((u_short)(n) >= 256 || (u_short)(n) == 0) { \
            *cp++ = 0; \
            cp[1] = (n); \
            cp[0] = (n) >> 8; \
            cp += 2; \
    } else { \
            *cp++ = (n); \
    } \
}

/*
* DECODEL takes the (compressed) change at byte cp and adds it to the
* current value of packet field 'f' (which must be a 4-byte (long) integer
* in network byte order).  DECODES does the same for a 2-byte (short) field.
* DECODEU takes the change at cp and stuffs it into the (short) field f.
* 'cp' is updated to point to the next field in the compressed header.
*/
#define DECODEL(f) { \
    if (*cp == 0) {\
            (f) = htonl(ntohl(f) + ((cp[1] << 8) | cp[2])); \
            cp += 3; \
    } else { \
            (f) = htonl(ntohl(f) + (u_long)*cp++); \
    } \
}
#define DECODES(f) { \
    if (*cp == 0) {\
            (f) = htons(ntohs(f) + ((cp[1] << 8) | cp[2])); \
            cp += 3; \
    } else { \
            (f) = htons(ntohs(f) + (u_long)*cp++); \
    } \
}
#define DECODEU(f) { \
    if (*cp == 0) {\
            (f) = htons((cp[1] << 8) | cp[2]); \
            cp += 3; \
    } else { \
            (f) = htons((u_long)*cp++); \
    } \
}


const u_char * sl_uncompress_tcp(u_char *bufp, int len, u_int type, struct slcompress *comp)
{
    u_char *cp;
    u_int hlen, changes;
    struct tcphdr *th;
    struct cstate *cs;
    struct ip *ip;

    switch (type) {

    case TYPE_ERROR:
    default:
            goto bad;

    case TYPE_IP:
            return (bufp);

    case TYPE_UNCOMPRESSED_TCP:
            /*
            * Locate the saved state for this connection.  If the state
            * index is legal, clear the 'discard' flag.
            */
            ip = (struct ip *) bufp;
            if (ip->ip_p >= MAX_STATES)
                goto bad;

            cs = &comp->rstate[comp->last_recv = ip->ip_p];
            comp->flags &= ~SLF_TOSS;
            /*
            * Restore the IP protocol field then save a copy of this
            * packet header.  (The checksum is zeroed in the copy so we
            * don't have to zero it each time we process a compressed
            * packet.
            */
            ip->ip_p = IPPROTO_TCP;
            hlen = ip->ip_hl;
            hlen += ((const struct tcphdr *)(&bufp[hlen<<2]))->th_off;
//            hlen += ((struct tcphdr *) & ((int *) ip)[hlen])->th_off;
            hlen <<= 2;
            memcpy(&cs->cs_ip, ip, hlen);
            cs->cs_ip.ip_sum = 0;
            cs->cs_hlen = hlen;
            return (bufp);

    case TYPE_COMPRESSED_TCP:
            break;
    }
    /* We've got a compressed packet. */
    cp = bufp;
    changes = *cp++;
    if (changes & NEW_C) {
            /*
            * Make sure the state index is in range, then grab the
            * state. If we have a good state index, clear the 'discard'
            * flag.
            */
            if (*cp >= MAX_STATES)
                goto bad;

            comp->flags &= ~SLF_TOSS;
            comp->last_recv = *cp++;
    } else {
            /*
            * This packet has an implicit state index.  If we've had a
            * line error since the last time we got an explicit state
            * index, we have to toss the packet.
            */
            if (comp->flags & SLF_TOSS)
                return ((u_char *) 0);
    }
    /*
        * Find the state then fill in the TCP checksum and PUSH bit.
        */
    cs = &comp->rstate[comp->last_recv];
    hlen = cs->cs_ip.ip_hl << 2;
    th = (struct tcphdr *) & ((u_char *) &cs->cs_ip)[hlen];
    th->th_sum = htons((*cp << 8) | cp[1]);
    cp += 2;
    if (changes & TCP_PUSH_BIT)
            th->th_flags |= TH_PUSH;
    else
            th->th_flags &= ~TH_PUSH;

    /*

        * Fix up the state's ack, seq, urg and win fields based on the
        * changemask.
        */
    switch (changes & SPECIALS_MASK) {
    case SPECIAL_I:
            {
            register u_int i = ntohs(cs->cs_ip.ip_len) - cs->cs_hlen;
            th->th_ack = htonl(ntohl(th->th_ack) + i);
            th->th_seq = htonl(ntohl(th->th_seq) + i);
            }
            break;

    case SPECIAL_D:
            th->th_seq = htonl(ntohl(th->th_seq) + ntohs(cs->cs_ip.ip_len)
                        - cs->cs_hlen);
            break;

    default:
            if (changes & NEW_U) {
                th->th_flags |= TH_URG;
                DECODEU(th->th_urp)
            } else
                th->th_flags &= ~TH_URG;
            if (changes & NEW_W)
                DECODES(th->th_win)
            if (changes & NEW_A)
                DECODEL(th->th_ack)
            if (changes & NEW_S)
                DECODEL(th->th_seq)
            break;
    }
    /* Update the IP ID */
    if (changes & NEW_I)
            DECODES(cs->cs_ip.ip_id)
    else
            cs->cs_ip.ip_id = htons(ntohs(cs->cs_ip.ip_id) + 1);

    /*
        * At this point, cp points to the first byte of data in the packet.
        * If we're not aligned on a 4-byte boundary, copy the data down so
        * the IP & TCP headers will be aligned.  Then back up cp by the
        * TCP/IP header length to make room for the reconstructed header (we
        * assume the packet we were handed has enough space to prepend 128
        * bytes of header).  Adjust the lenth to account for the new header
        * & fill in the IP total length.
        */
    len -= (int)(cp - bufp);
    if (len < 0)
            /*
            * we must have dropped some characters (crc should detect
            * this but the old slip framing won't)
            */
            goto bad;

    // Make length from IP-TCP header
    hlen = cs->cs_ip.ip_hl;                                             // IP-Size (in 32bit)
    hlen += ((const struct tcphdr *)(&cs->cs_hdr[hlen<<2]))->th_off;    // TCP-Size(in 32bit)
    hlen <<= 2;                                                         // Make to bytes

    if ( len )
    { // Move data to end of IP header
        memmove((u_char *)&bufp[hlen], (u_char*)cp, len);
    }

    // Make IP-Header (data-length + IP+TCP-Header length)
    cs->cs_ip.ip_len = htons( hlen + len );
    { // recompute the ip header checksum
        cs->cs_ip.ip_sum = 0; len = (cs->cs_ip.ip_hl << 2);
        register u_short *bp = (u_short *)&cs->cs_ip;
        for (changes = 0; len > 0; len -= 2)
            changes += *bp++;
        changes = (changes & 0xffff) + (changes >> 16);
        changes = (changes & 0xffff) + (changes >> 16);
        cs->cs_ip.ip_sum = ~changes;
    }
    // Copy header
    memcpy(bufp, &cs->cs_ip, hlen);

    return bufp;
#if 0
    if ((INT_PTR) cp & 3) {
            if (len > 0)
                memmove((u_char *)((INT_PTR) cp & ~3), (u_char*)cp, len);
            cp = (u_char *) ((INT_PTR) cp & ~3);
    }
    cp -= cs->cs_hlen;
    len += cs->cs_hlen;
    cs->cs_ip.ip_len = htons(len);
    memcpy(cp, &cs->cs_ip, cs->cs_hlen);

    /* recompute the ip header checksum */
    {
            register u_short *bp = (u_short *) cp;
            for (changes = 0; hlen > 0; hlen -= 2)
                changes += *bp++;
            changes = (changes & 0xffff) + (changes >> 16);
            changes = (changes & 0xffff) + (changes >> 16);
            ((struct ip *) cp)->ip_sum = ~changes;
    }
    return (cp);
#endif // 0

bad:
    comp->flags |= SLF_TOSS;
    return ((u_char *) 0);
}

void sl_compress_init( struct slcompress *comp )
{
    register u_int i;
    register struct cstate *tstate = comp->tstate;

    /*
        * Clean out any junk left from the last time line was used.
        */
    memset((char *) comp, 0, sizeof(*comp));
    /*
        * Link the transmit states into a circular list.
        */
    for (i = MAX_STATES - 1; i > 0; --i) {
            tstate[i].cs_id = i;
            tstate[i].cs_next = &tstate[i - 1];
    }
    tstate[0].cs_next = &tstate[MAX_STATES - 1];
    tstate[0].cs_id = 0;
    comp->last_cs = &tstate[0];
    /*
        * Make sure we don't accidentally do CID compression
        * (assumes MAX_STATES < 255).
        */
    comp->last_recv = 255;
    comp->last_xmit = 255;
}


#ifndef ARRAY_SIZE
    #define ARRAY_SIZE( sArray )                (sizeof((sArray))/sizeof((sArray)[0]))
#endif // ARRAY_SIZE

struct StructCompressionState
{
    struct slcompress   stEndPoint[2];  // Uplink/Downlink direction
};

CRFC1144::CRFC1144()
{
    m_pstState  = new StructCompressionState;
    if ( m_pstState )
    { // Clean all
        memset(m_pstState, 0, sizeof(m_pstState) );

        for ( unsigned char ucState = 0; ARRAY_SIZE(m_pstState->stEndPoint) > ucState; ucState++ )
        {   sl_compress_init( &m_pstState->stEndPoint[ucState] );   }
    }
}

CRFC1144::~CRFC1144()
{
    if ( m_pstState )
    {   delete m_pstState; m_pstState = NULL;   }
}

void CRFC1144::Init( const unsigned char ucDirection )
{
    if ( ARRAY_SIZE(m_pstState->stEndPoint) > ucDirection )
    {   sl_compress_init( &m_pstState->stEndPoint[ucDirection] );   }
}

// Decompress data
const unsigned char *CRFC1144::Decompress( unsigned char *pucData, const int lLength, const unsigned char ucDirection, bool bCompressed )
{
    if ( ARRAY_SIZE(m_pstState->stEndPoint) <= ucDirection )
    {   return NULL;    }

    // Send data
    return sl_uncompress_tcp(pucData, lLength, (bCompressed) ? (TYPE_COMPRESSED_TCP) : (TYPE_UNCOMPRESSED_TCP), &m_pstState->stEndPoint[ucDirection] );

}
